<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSRMASync\Test\Unit\Sync;

use Magento\SaaSOrderSync\Api\OrderSync\Bulk\OperationData;
use Magento\SaaSOrderSync\Api\Result;
use Magento\SaaSOrderSync\Core\OrderSync\SaaSClient;
use Magento\SaaSOrderSync\Test\Unit\Core\SaaS\SaaSClientMockResolver;
use Magento\SaaSRMASync\Sync\AggregateExporter;
use Magento\SaaSRMASync\Sync\AggregateMapper;
use Magento\SaaSRMASync\Sync\RMARepository;
use Magento\SaaSRMASync\Sync\SyncRMAAggregatesOperationProcessor;
use function PHPUnit\Framework\anything;
use function PHPUnit\Framework\assertThat;
use function PHPUnit\Framework\assertTrue;
use function PHPUnit\Framework\equalTo;
use PHPUnit\Framework\MockObject\MockObject;
use function PHPUnit\Framework\never;
use function PHPUnit\Framework\once;
use PHPUnit\Framework\TestCase;

class SyncRMAAggregatesOperationProcessorTest extends TestCase
{
    private SaaSClient&MockObject $orderSyncSaaSClientMock;
    private RMARepository&MockObject $rmaRepositoryMock;

    private SyncRMAAggregatesOperationProcessor $syncRMAAggregatesOperationProcessor;

    public function setUp(): void
    {
        $clientResolver = new SaaSClientMockResolver($this);
        $this->orderSyncSaaSClientMock = $clientResolver->createOrderSyncClient();
        $this->rmaRepositoryMock = $this->createMock(RMARepository::class);

        $this->syncRMAAggregatesOperationProcessor = new SyncRMAAggregatesOperationProcessor(
            $clientResolver,
            new AggregateExporter($this->rmaRepositoryMock),
            new AggregateMapper(),
        );
    }

    public function test_shouldReturnError_whenRMATablesAreNotAvailable()
    {
        $syncId = __METHOD__;
        $operationData = new OperationData();
        $operationData->setOrderIds([1, 2, 3, 4, 5]);

        $this->rmaRepositoryMock->expects(once())
            ->method('rmaTablesAvailable')
            ->willReturn(false);

        $this->orderSyncSaaSClientMock->expects(never())
            ->method(anything());

        $result = $this->syncRMAAggregatesOperationProcessor->process($syncId, $operationData->getOrderIds());

        assertTrue($result->isError());
        assertThat($result->getErrorCode(), equalTo('RMA_TABLES_NOT_FOUND'));
    }

    public function test_shouldReturnData_whenNoRMAAggregates()
    {
        $syncId = __METHOD__;
        $operationData = new OperationData();
        $operationData->setOrderIds([1, 2, 3, 4, 5]);

        $this->rmaRepositoryMock->expects(once())
            ->method('rmaTablesAvailable')
            ->willReturn(true);

        $this->rmaRepositoryMock->expects(once())
            ->method('findRMAAggregates')
            ->with($operationData->getOrderIds())
            ->willReturn([]);

        $this->orderSyncSaaSClientMock->expects(never())
            ->method(anything());

        $result = $this->syncRMAAggregatesOperationProcessor->process($syncId, $operationData->getOrderIds());

        assertTrue($result->isData());
        [, $data] = $result;
        assertThat($data['code'], equalTo("EMPTY_RMA_AGGREGATES"));
    }

    public function test_shouldReturnData_whenRMAAggregatesArePushed()
    {
        $syncId = __METHOD__;
        $operationData = new OperationData();
        $operationData->setOrderIds([1, 2, 3, 4, 5]);

        $this->rmaRepositoryMock->expects(once())
            ->method('rmaTablesAvailable')
            ->willReturn(true);

        $this->rmaRepositoryMock->expects(once())
            ->method('findRMAAggregates')
            ->with($operationData->getOrderIds())
            ->willReturn([
                $this->rmaRecord('1', '2', '100', '00100', '52ED964E-788C-404D-8101-ED407AE613AD', '2023-01-01 00:00:00', 'closed', '2023-01-01 00:00:00'),
                $this->rmaRecord('2', '3', '101', '00101', 'D96C2FD2-3542-49DD-82EE-F121BDA219A6', '2023-02-01 00:00:00', 'denied', '2023-02-02 00:00:00'),
                $this->rmaRecord('3', '4', '102', '00102', '7B860C61-6CF3-412B-A65C-9E00CBF51ABB', '2023-03-01 00:00:00', 'approved', '2023-03-03 00:00:00'),
            ]);

        $this->rmaRepositoryMock->expects(once())
            ->method('findRMAItems')
            ->with([1, 2, 3])
            ->willReturn([
                $this->rmaItemRecord('1', '100', '200', 'closed', '2', '1', '1', '2'),
                $this->rmaItemRecord('1', '101', '201', 'closed', '2', '1', '2', '1'),
                $this->rmaItemRecord('2', '102', '202', 'rejected_on_item', '1', null, null, null),
                $this->rmaItemRecord('3', '103', '203', 'approved_on_item', '2', '2', null, '2'),
                $this->rmaItemRecord('3', '104', '204', 'received_on_item', '1', '1', '1', '1'),
            ]);

        $this->rmaRepositoryMock->expects(once())
            ->method('findRMAItemOptions')
            ->with([100, 101, 102, 103, 104])
            ->willReturn([
                $this->rmaItemOptionRecord('100', 'Resolution', 'Refund'),
                $this->rmaItemOptionRecord('100', 'Status', 'Unopened'),
                $this->rmaItemOptionRecord('101', 'Resolution', 'Credit'),
                $this->rmaItemOptionRecord('101', 'Status', 'Opened'),
            ]);

        $saasResponse = [
            'syncId' => $syncId,
            'processedCount' => 3,
            'errorCount' => 0,
        ];
        $this->orderSyncSaaSClientMock->expects(once())
            ->method('syncRMABatch')
            ->with($syncId, [
                $this->rmaAggregate(
                    '1',
                    '2',
                    '100',
                    '00100',
                    '52ED964E-788C-404D-8101-ED407AE613AD',
                    '2023-01-01T00:00:00+00:00',
                    11,
                    '2023-01-01T00:00:00+00:00',
                    [
                        $this->rmaItemAggregate(
                            '200',
                            11,
                            2,
                            1,
                            1,
                            2,
                            [['label' => 'Resolution', 'value' => 'Refund'], ['label' => 'Status', 'value' => 'Unopened']]
                        ),
                        $this->rmaItemAggregate(
                            '201',
                            11,
                            2,
                            1,
                            2,
                            1,
                            [['label' => 'Resolution', 'value' => 'Credit'], ['label' => 'Status', 'value' => 'Opened']]
                        ),
                    ]
                ),
                $this->rmaAggregate(
                    '2',
                    '3',
                    '101',
                    '00101',
                    'D96C2FD2-3542-49DD-82EE-F121BDA219A6',
                    '2023-02-01T00:00:00+00:00',
                    10,
                    '2023-02-02T00:00:00+00:00',
                    [
                        $this->rmaItemAggregate('202', 9, 1),
                    ]
                ),
                $this->rmaAggregate(
                    '3',
                    '4',
                    '102',
                    '00102',
                    '7B860C61-6CF3-412B-A65C-9E00CBF51ABB',
                    '2023-03-01T00:00:00+00:00',
                    6,
                    '2023-03-03T00:00:00+00:00',
                    [
                        $this->rmaItemAggregate('203', 7, 2, 2, null, 2),
                        $this->rmaItemAggregate('204', 5, 1, 1, 1, 1),
                    ]
                ),
            ])
            ->willReturn(Result::data($saasResponse));

        $result = $this->syncRMAAggregatesOperationProcessor->process($syncId, $operationData->getOrderIds());

        assertTrue($result->isData());
        [, $data] = $result;
        assertThat($data, equalTo($saasResponse));
    }

    private function rmaRecord(string $entityId, string $incrementId, string $orderId, string $orderIncrementId, string $uuid, string $dateRequested, string $status, string $statusUpdatedAt)
    {
        return [
            'entity_id' => $entityId,
            'increment_id' => $incrementId,
            'order_id' => $orderId,
            'order_increment_id' => $orderIncrementId,
            'uuid' => $uuid,
            'date_requested' => $dateRequested,
            'status' => $status,
            'status_updated_at' => $statusUpdatedAt,
        ];
    }

    private function rmaItemRecord(
        string $rmaEntityId,
        string $entityId,
        string $orderItemId,
        string $status,
        string $qtyRequested,
        ?string $qtyAuthorized,
        ?string $qtyReturned,
        ?string $qtyApproved
    ) {
        return [
            'rma_entity_id' => $rmaEntityId,
            'entity_id' => $entityId,
            'order_item_id' => $orderItemId,
            'status' => $status,
            'qty_requested' => $qtyRequested,
            'qty_authorized' => $qtyAuthorized,
            'qty_returned' => $qtyReturned,
            'qty_approved' => $qtyApproved,
        ];
    }

    private function rmaItemOptionRecord(string $rmaItemId, string $label, string $value)
    {
        return [
            'entity_id' => $rmaItemId,
            'label' => $label,
            'value' => $value,
        ];
    }

    private function rmaAggregate(string $rmaId, string $rmaNumber, string $orderId, string $orderNumber, string $orderUuid, string $requestedAt, int $status, string $statusUpdatedAt, array $rmaItems)
    {
        return [
            'rmaId' => $rmaId,
            'rmaNumber' => $rmaNumber,
            'orderId' => $orderId,
            'orderNumber' => $orderNumber,
            'orderUuid' => $orderUuid,
            'requestedAt' => $requestedAt,
            'status' => $status,
            'statusUpdatedAt' => $statusUpdatedAt,
            'rmaItems' => $rmaItems,
        ];
    }

    private function rmaItemAggregate(
        string $orderItemId,
        int $status,
        float $qtyRequested,
        ?float $qtyAuthorized = null,
        ?float $qtyReturned = null,
        ?float $qtyApproved = null,
        ?array $options = null
    ) {
        $data = [
            'orderItemId' => $orderItemId,
            'status' => $status,
            'qtyRequested' => $qtyRequested,
        ];
        if ($qtyAuthorized) {
            $data['qtyAuthorized'] = $qtyAuthorized;
        }
        if ($qtyReturned) {
            $data['qtyReturned'] = $qtyReturned;
        }
        if ($qtyApproved) {
            $data['qtyApproved'] = $qtyApproved;
        }
        if ($options) {
            $data['options'] = $options;
        }

        return $data;
    }
}
