<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSRMASync\Sync;

use Magento\SaaSOrderSync\Api\OrderSync\Bulk\OperationData;
use Magento\SaaSOrderSync\Api\OrderSync\Bulk\OperationProcessor;
use Magento\SaaSOrderSync\Api\Result;
use Magento\SaaSOrderSync\Core\SaaS\SaaSClientResolverInterface;

/**
 * RMA Aggregates processing is injected in Order Sync process through OperationProcessorPool
 *
 * @see ../etc/di.xml
 */
class SyncRMAAggregatesOperationProcessor implements OperationProcessor
{
    private SaaSClientResolverInterface $clientResolver;
    private AggregateExporter $rmaAggregatesExporter;

    private AggregateMapper $mapper;

    public function __construct(
        SaaSClientResolverInterface $clientResolver,
        AggregateExporter           $rmaAggregatesExporter,
        AggregateMapper             $mapper,
    ) {
        $this->clientResolver = $clientResolver;
        $this->rmaAggregatesExporter = $rmaAggregatesExporter;
        $this->mapper = $mapper;
    }

    public function process(string $syncId, array $orderIds): Result
    {
        return Result::wrap(function () use ($syncId, $orderIds) {
            $result = $this->rmaAggregatesExporter->export($orderIds, [$this->mapper, 'map']);
            if ($result->isError()) {
                return $result;
            }

            [, $rmaAggregates] = $result;
            if (empty($rmaAggregates)) {
                return Result::data(['code' => 'EMPTY_RMA_AGGREGATES']);
            }

            $orderSyncClient = $this->clientResolver->createOrderSyncClient();
            return $orderSyncClient->syncRMABatch($syncId, $rmaAggregates);
        });
    }

    public function order(): int
    {
        return OperationProcessor::HIGHEST_PRECEDENCE + 10;
    }
}
