<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSRMASync\Sync;

use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\TableNotFoundException;
use Zend_Db_Expr;

class RMARepository
{

    private ResourceConnection $resourceConnection;

    public function __construct(ResourceConnection $resourceConnection)
    {
        $this->resourceConnection = $resourceConnection;
    }

    public function rmaTablesAvailable(): bool
    {
        return $this->allTablesExist(
            $this->rmaTableName(),
            $this->rmaStatusHistoryTableName(),
            $this->rmaItemEntityTableName(),
            $this->rmaItemEntityIntTableName(),
            $this->attributeTableName(),
            $this->attributeOptionValueTableName(),
        );
    }

    public function findRMAAggregates(array $orderIds): array
    {
        $connection = $this->resourceConnection->getConnection();

        $selectStatusHistory = $connection->select()
            ->from(
                $this->rmaStatusHistoryTableName(),
                ['rma_entity_id', 'MAX(created_at) AS status_updated_at'],
            )
            ->group('rma_entity_id');

        $selectAggregates = $connection->select()
            ->from(
                ['r' => $this->rmaTableName()],
                [
                    'r.entity_id',
                    'r.increment_id',
                    'r.order_id',
                    'r.order_increment_id',
                    'r.date_requested',
                    'r.status',
                    'h.status_updated_at',
                    'u.uuid',
                ]
            )
            ->joinLeft(['h' => $selectStatusHistory], 'r.entity_id=h.rma_entity_id', [])
            ->joinLeft(['u' => $this->dataExporterUuidTable()], 'r.order_id = u.entity_id AND u.type=\'order\'', [])
            ->where('order_id IN (?)', $orderIds);

        return $connection->fetchAll($selectAggregates);
    }

    public function findRMAItems(array $rmaIds): array
    {
        $connection = $this->resourceConnection->getConnection();

        $selectAggregates = $connection->select()
            ->from(
                $this->rmaItemEntityTableName(),
                [
                    'entity_id',
                    'rma_entity_id',
                    'order_item_id',
                    'status',
                    'qty_requested',
                    'qty_authorized',
                    'qty_returned',
                    'qty_approved',
                ]
            )
            ->where('rma_entity_id IN (?)', $rmaIds);

        return $connection->fetchAll($selectAggregates);
    }

    public function findRecentClosedRMADate(string $rmaId): string
    {
        $connection = $this->resourceConnection->getConnection();

        $selectDate = $connection->select()
            ->from(
                $this->rmaStatusHistoryTableName(),
                [
                    'created_at',
                ]
            )
            ->where('status="closed" and rma_entity_id = (?)', $rmaId)
            ->order('created_at', 'DESC')
            ->limit(1);

        return $connection->fetchOne($selectDate);
    }

    public function findRMAItemOptions(array $rmaItemIds): array
    {
        $connection = $this->resourceConnection->getConnection();

        $selectAggregates = $connection->select()
            ->from(
                ['ri' => $this->rmaItemEntityIntTableName()],
                [
                    'ri.entity_id',
                    'ri.attribute_id',
                    'value_id' => 'ri.value',
                    'value' => 'ao.value',
                    'label' => 'a.frontend_label'
                ]
            )
            ->joinLeft(
                ['a' => $this->attributeTableName()],
                'ri.attribute_id = a.attribute_id',
                []
            )
            ->joinLeft(
                ['ao' => $this->attributeOptionValueTableName()],
                'ri.value = ao.value_id',
                []
            )
            ->where('entity_id IN (?)', $rmaItemIds);

        return $connection->fetchAll($selectAggregates);
    }

    public function findOrderUuid($order_entity_id): ?string
    {
        $connection = $this->resourceConnection->getConnection();

        $selectUuid = $connection->select()
            ->from(
                $this->dataExporterUuidTable(),
                ['uuid'],
            )
            ->where('type = ? ', 'order')
            ->where('entity_id = ? ', $order_entity_id);

        $row = $connection->fetchRow($selectUuid) ?? [];
        return $row['uuid'];
    }

    private function dataExporterUuidTable(): string
    {
        return $this->resourceConnection->getTableName('data_exporter_uuid');
    }

    private function rmaTableName(): string
    {
        return $this->resourceConnection->getTableName('magento_rma');
    }

    private function rmaStatusHistoryTableName(): string
    {
        return $this->resourceConnection->getTableName('magento_rma_status_history');
    }

    private function rmaItemEntityTableName(): string
    {
        return $this->resourceConnection->getTableName('magento_rma_item_entity');
    }

    private function rmaItemEntityIntTableName(): string
    {
        return $this->resourceConnection->getTableName('magento_rma_item_entity_int');
    }

    private function attributeTableName(): string
    {
        return $this->resourceConnection->getTableName('eav_attribute');
    }

    private function attributeOptionValueTableName(): string
    {
        return $this->resourceConnection->getTableName('eav_attribute_option_value');
    }

    private function allTablesExist(...$tableNames): bool
    {
        try {
            $connection = $this->resourceConnection->getConnection();
            $selects = array_map(
                function ($tableName) use ($connection) {
                    $select = $connection->select()
                        ->from($tableName, [new Zend_Db_Expr($connection->quote('1'))])
                        ->limit(1);
                    return "($select)";
                },
                $tableNames
            );
            $union = $connection->select()->union($selects);
            $connection->fetchCol($union);

            return true;
        } catch (TableNotFoundException $e) {
            return false;
        }
    }
}
