<?php
declare(strict_types=1);

namespace Magento\SaaSRMASync\Sync;

use DateTime;
use DateTimeInterface;
use DateTimeZone;

/**
 * Maps into the required format of the proto specification.
 *
 * @link https://git.corp.adobe.com/magento/proto-schemas/blob/main/order/src/main/proto/com/adobe/commerce/sales/order/aggregate/v1/commerce_rma_aggregate.proto CommerceRMAAggregate proto schema
 */
class AggregateMapper
{
    public function map(array $rmaRecord): array
    {
        return [
            'rmaId' => $rmaRecord['entity_id'],
            'orderId' => $rmaRecord['order_id'],
            'orderNumber' => $rmaRecord['order_increment_id'],
            'requestedAt' => $this->mapToIso8601($rmaRecord['date_requested']),
            'status' => $this->mapStatus($rmaRecord['status']),
            'statusUpdatedAt' => $this->mapToIso8601($rmaRecord['status_updated_at']),
            'rmaItems' => array_map([$this, 'mapItem'], $rmaRecord['rma_items']),
        ];
    }

    private function mapItem(array $itemRecord): array
    {
        $rmaItem = [
            'orderItemId' => $itemRecord['order_item_id'],
            'status' => $this->mapStatus($itemRecord['status']),
            'qtyRequested' => $this->mapToFloat($itemRecord['qty_requested']),
        ];
        if ($itemRecord['qty_authorized']) {
            $rmaItem['qtyAuthorized'] = $this->mapToFloat($itemRecord['qty_authorized']);
        }
        if ($itemRecord['qty_returned']) {
            $rmaItem['qtyReturned'] = $this->mapToFloat($itemRecord['qty_returned']);
        }
        if ($itemRecord['qty_approved']) {
            $rmaItem['qtyApproved'] = $this->mapToFloat($itemRecord['qty_approved']);
        }
        if ($itemRecord['options']) {
            $rmaItem['options'] = array_map([$this, 'mapOption'], $itemRecord['options']);
        }

        return $rmaItem;
    }

    private function mapOption(array $itemOptionRecord): array
    {
        return [
            'label' => $itemOptionRecord['label'],
            'value' => $itemOptionRecord['value'],
        ];
    }

    private function mapStatus(?string $status): int
    {
        return match (strtoupper($status)) {
            'PENDING' => 1,
            'AUTHORIZED' => 2,
            'PARTIALLY_AUTHORIZED' => 3,
            'RECEIVED' => 4,
            'RECEIVED_ON_ITEM' => 5,
            'APPROVED' => 6,
            'APPROVED_ON_ITEM' => 7,
            'REJECTED' => 8,
            'REJECTED_ON_ITEM' => 9,
            'DENIED' => 10,
            'CLOSED' => 11,
            'PROCESSED_CLOSED' => 12,
            default => 0,
        };
    }

    private function mapToFloat(?string $value): float
    {
        if ($value) {
            return floatval($value);
        }
        return 0.0;
    }

    private function mapToIso8601(string $dateTime): string
    {
        return (new DateTime($dateTime, new DateTimeZone('UTC')))->format(DateTimeInterface::ATOM);
    }
}
