<?php
declare(strict_types=1);

namespace Magento\SaaSRMASync\Sync;

use Magento\SaaSOrderSync\Api\Result;

class AggregateExporter
{
    private RMARepository $rmaRepository;

    public function __construct(RMARepository $rmaRepository)
    {
        $this->rmaRepository = $rmaRepository;
    }

    public function export(array $orderIds, callable $mapper): Result
    {
        if (!$this->rmaRepository->rmaTablesAvailable()) {
            return Result::error('RMA_TABLES_NOT_FOUND');
        }

        $aggregates = $this->rmaRepository->findRMAAggregates($orderIds);
        if (empty($aggregates)) {
            return Result::data([]);
        }

        $aggregateItems = $this->getRMAItemsForAggregates($aggregates);

        return Result::data($this->combine($aggregates, $aggregateItems, $mapper));
    }

    private function getRMAItemsForAggregates(array $aggregates): array
    {
        $rmaIds = array_map(fn ($rma) => $rma['entity_id'], $aggregates);
        $aggregateItems = $this->rmaRepository->findRMAItems($rmaIds);

        $itemIds = array_map(fn ($rmaItem) => $rmaItem['entity_id'], $aggregateItems);
        $itemOptions = $this->rmaRepository->findRMAItemOptions($itemIds);

        $itemOptionsGroupedByItemId = array_reduce($itemOptions, $this->groupBy('entity_id'), []);
        foreach ($aggregateItems as &$rmaItem) {
            $rmaItem['options'] = $itemOptionsGroupedByItemId[$rmaItem['entity_id']] ?? [];
        }

        return $aggregateItems;
    }

    private function combine(array $rmaAggregates, array $rmaAggregatesItems, callable $mapper): array
    {
        $itemsGroupedByRmaId = array_reduce($rmaAggregatesItems, $this->groupBy('rma_entity_id'), []);
        return array_map($this->map($itemsGroupedByRmaId, $mapper), $rmaAggregates);
    }

    public function map(array $itemsGroupedByRmaId, callable $mapper): callable
    {
        return function ($rmaRecord) use ($mapper, $itemsGroupedByRmaId) {
            $rmaRecord['rma_items'] = $itemsGroupedByRmaId[$rmaRecord['entity_id']] ?? [];

            return call_user_func($mapper, $rmaRecord);
        };
    }

    private function groupBy(string $keyName): callable
    {
        return function (array $accumulator, array $element) use ($keyName) {
            $accumulator[$element[$keyName]][] = $element;
            return $accumulator;
        };
    }
}
