<?php
declare(strict_types=1);

namespace Magento\SaaSRMASync\Sync;

use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Adapter\TableNotFoundException;
use Zend_Db_Expr;

class RMARepository
{

    private ResourceConnection $resourceConnection;

    public function __construct(ResourceConnection $resourceConnection)
    {
        $this->resourceConnection = $resourceConnection;
    }

    public function rmaTablesAvailable(): bool
    {
        return $this->allTablesExist(
            $this->rmaTableName(),
            $this->rmaStatusHistoryTableName(),
            $this->rmaItemEntityTableName(),
        );
    }

    public function findRMAAggregates(array $orderIds): array
    {
        $connection = $this->resourceConnection->getConnection();

        $selectStatusHistory = $connection->select()
            ->from(
                $this->rmaStatusHistoryTableName(),
                ['rma_entity_id', 'MAX(created_at) AS status_updated_at'],
            )
            ->group('rma_entity_id');

        $selectAggregates = $connection->select()
            ->from(
                ['r' => $this->rmaTableName()],
                [
                    'r.entity_id',
                    'r.order_id',
                    'r.order_increment_id',
                    'r.date_requested',
                    'r.status',
                    'h.status_updated_at',
                ]
            )
            ->joinLeft(['h' => $selectStatusHistory], 'r.entity_id=h.rma_entity_id', [])
            ->where('order_id IN (?)', $orderIds);

        return $connection->fetchAll($selectAggregates);
    }

    public function findRMAItems(array $rmaIds): array
    {
        $connection = $this->resourceConnection->getConnection();

        $selectAggregates = $connection->select()
            ->from(
                $this->rmaItemEntityTableName(),
                [
                    'rma_entity_id',
                    'order_item_id',
                    'status',
                    'qty_requested',
                    'qty_authorized',
                    'qty_returned',
                    'qty_approved',
                ]
            )
            ->where('rma_entity_id IN (?)', $rmaIds);

        return $connection->fetchAll($selectAggregates);
    }

    private function rmaTableName(): string
    {
        return $this->resourceConnection->getTableName('magento_rma');
    }

    private function rmaStatusHistoryTableName(): string
    {
        return $this->resourceConnection->getTableName('magento_rma_status_history');
    }

    private function rmaItemEntityTableName(): string
    {
        return $this->resourceConnection->getTableName('magento_rma_item_entity');
    }

    private function allTablesExist(...$tableNames): bool
    {
        try {
            $connection = $this->resourceConnection->getConnection();
            $selects = array_map(
                function ($tableName) use ($connection) {
                    $select = $connection->select()
                        ->from($tableName, [new Zend_Db_Expr($connection->quote('1'))])
                        ->limit(1);
                    return "($select)";
                },
                $tableNames
            );
            $union = $connection->select()->union($selects);
            $connection->fetchCol($union);

            return true;
        } catch (TableNotFoundException $e) {
            return false;
        }
    }
}
