<?php
declare(strict_types=1);

use Magento\SaaSOrderSync\Api\OrderSync\Bulk\OperationData;
use Magento\SaaSOrderSync\Api\Result;
use Magento\SaaSOrderSync\Core\OrderSync\SaaSClient;
use Magento\SaaSOrderSync\Test\Unit\Core\SaaS\SaaSClientMockResolver;
use Magento\SaaSRMASync\Sync\AggregateExporter;
use Magento\SaaSRMASync\Sync\AggregateMapper;
use Magento\SaaSRMASync\Sync\RMARepository;
use Magento\SaaSRMASync\Sync\SyncRMAAggregatesOperationProcessor;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use function PHPUnit\Framework\anything;
use function PHPUnit\Framework\assertThat;
use function PHPUnit\Framework\assertTrue;
use function PHPUnit\Framework\equalTo;
use function PHPUnit\Framework\never;
use function PHPUnit\Framework\once;

class SyncRMAAggregatesOperationProcessorTest extends TestCase
{
    private SaaSClient&MockObject $orderSyncSaaSClientMock;
    private RMARepository&MockObject $rmaRepositoryMock;

    private SyncRMAAggregatesOperationProcessor $syncRMAAggregatesOperationProcessor;

    public function setUp(): void
    {
        $clientResolver = new SaaSClientMockResolver($this);
        $this->orderSyncSaaSClientMock = $clientResolver->createOrderSyncClient();
        $this->rmaRepositoryMock = $this->createMock(RMARepository::class);

        $this->syncRMAAggregatesOperationProcessor = new SyncRMAAggregatesOperationProcessor(
            $clientResolver,
            new AggregateExporter($this->rmaRepositoryMock),
            new AggregateMapper(),
        );
    }

    public function test_shouldReturnError_whenRMATablesAreNotAvailable()
    {
        $syncId = __METHOD__;
        $operationData = new OperationData();
        $operationData->setOrderIds([1, 2, 3, 4, 5]);

        $this->rmaRepositoryMock->expects(once())
            ->method('rmaTablesAvailable')
            ->willReturn(false);

        $this->orderSyncSaaSClientMock->expects(never())
            ->method(anything());

        $result = $this->syncRMAAggregatesOperationProcessor->process($syncId, $operationData);

        assertTrue($result->isError());
        assertThat($result->getErrorCode(), equalTo('RMA_TABLES_NOT_FOUND'));
    }

    public function test_shouldReturnData_whenNoRMAAggregates()
    {
        $syncId = __METHOD__;
        $operationData = new OperationData();
        $operationData->setOrderIds([1, 2, 3, 4, 5]);

        $this->rmaRepositoryMock->expects(once())
            ->method('rmaTablesAvailable')
            ->willReturn(true);

        $this->rmaRepositoryMock->expects(once())
            ->method('findRMAAggregates')
            ->with($operationData->getOrderIds())
            ->willReturn([]);

        $this->orderSyncSaaSClientMock->expects(never())
            ->method(anything());

        $result = $this->syncRMAAggregatesOperationProcessor->process($syncId, $operationData);

        assertTrue($result->isData());
        [, $data] = $result;
        assertThat($data['code'], equalTo("EMPTY_RMA_AGGREGATES"));
    }

    public function test_shouldReturnData_whenRMAAggregatesArePushed()
    {
        $syncId = __METHOD__;
        $operationData = new OperationData();
        $operationData->setOrderIds([1, 2, 3, 4, 5]);

        $this->rmaRepositoryMock->expects(once())
            ->method('rmaTablesAvailable')
            ->willReturn(true);

        $this->rmaRepositoryMock->expects(once())
            ->method('findRMAAggregates')
            ->with($operationData->getOrderIds())
            ->willReturn([
                $this->rmaRecord('1', '100', '00100', '2023-01-01 00:00:00', 'closed', '2023-01-01 00:00:00'),
                $this->rmaRecord('2', '101', '00101', '2023-02-01 00:00:00', 'denied', '2023-02-02 00:00:00'),
                $this->rmaRecord('3', '102', '00102', '2023-03-01 00:00:00', 'approved', '2023-03-03 00:00:00'),
            ]);

        $this->rmaRepositoryMock->expects(once())
            ->method('findRMAItems')
            ->with([1, 2, 3])
            ->willReturn([
                $this->rmaItemRecord('1', '200', 'closed', '2', '1', '1', '2'),
                $this->rmaItemRecord('1', '201', 'closed', '2', '1', '2', '1'),
                $this->rmaItemRecord('2', '202', 'rejected_on_item', '1', null, null, null),
                $this->rmaItemRecord('3', '203', 'approved_on_item', '2', '2', null, '2'),
                $this->rmaItemRecord('3', '204', 'received_on_item', '1', '1', '1', '1'),
            ]);

        $saasResponse = [
            'syncId' => $syncId,
            'processedCount' => 3,
            'errorCount' => 0,
        ];
        $this->orderSyncSaaSClientMock->expects(once())
            ->method('syncRMABatch')
            ->with($syncId, [
                $this->rmaAggregate(
                    '1',
                    '100',
                    '00100',
                    '2023-01-01T00:00:00+00:00',
                    11,
                    '2023-01-01T00:00:00+00:00',
                    [
                        $this->rmaItemAggregate('200', 11, 2, 1, 1, 2),
                        $this->rmaItemAggregate('201', 11, 2, 1, 2, 1),
                    ]
                ),
                $this->rmaAggregate(
                    '2',
                    '101',
                    '00101',
                    '2023-02-01T00:00:00+00:00',
                    10,
                    '2023-02-02T00:00:00+00:00',
                    [
                        $this->rmaItemAggregate('202', 9, 1),
                    ]
                ),
                $this->rmaAggregate(
                    '3',
                    '102',
                    '00102',
                    '2023-03-01T00:00:00+00:00',
                    6,
                    '2023-03-03T00:00:00+00:00',
                    [
                        $this->rmaItemAggregate('203', 7, 2, 2, null, 2),
                        $this->rmaItemAggregate('204', 5, 1, 1, 1, 1),
                    ]
                ),
            ])
            ->willReturn(Result::data($saasResponse));

        $result = $this->syncRMAAggregatesOperationProcessor->process($syncId, $operationData);
        assertTrue($result->isData());
        [, $data] = $result;
        assertThat($data, equalTo($saasResponse));
    }

    private function rmaRecord(string $entityId, string $orderId, string $orderIncrementId, string $dateRequested, string $status, string $statusUpdatedAt)
    {
        return [
            'entity_id' => $entityId,
            'order_id' => $orderId,
            'order_increment_id' => $orderIncrementId,
            'date_requested' => $dateRequested,
            'status' => $status,
            'status_updated_at' => $statusUpdatedAt,
        ];
    }

    private function rmaItemRecord(string $rmaEntityId, string $orderItemId, string $status, string $qtyRequested, ?string $qtyAuthorized, ?string $qtyReturned, ?string $qtyApproved)
    {
        return [
            'rma_entity_id' => $rmaEntityId,
            'order_item_id' => $orderItemId,
            'status' => $status,
            'qty_requested' => $qtyRequested,
            'qty_authorized' => $qtyAuthorized,
            'qty_returned' => $qtyReturned,
            'qty_approved' => $qtyApproved,
        ];
    }

    private function rmaAggregate(string $rmaId, string $orderId, string $orderNumber, string $requestedAt, int $status, string $statusUpdatedAt, array $rmaItems)
    {
        return [
            'rmaId' => $rmaId,
            'orderId' => $orderId,
            'orderNumber' => $orderNumber,
            'requestedAt' => $requestedAt,
            'status' => $status,
            'statusUpdatedAt' => $statusUpdatedAt,
            'rmaItems' => $rmaItems,
        ];
    }

    private function rmaItemAggregate(string $orderItemId, int $status, float $qtyRequested, ?float $qtyAuthorized = null, ?float $qtyReturned = null, ?float $qtyApproved = null)
    {
        $data = [
            'orderItemId' => $orderItemId,
            'status' => $status,
            'qtyRequested' => $qtyRequested,
        ];
        if ($qtyAuthorized) {
            $data['qtyAuthorized'] = $qtyAuthorized;
        }
        if ($qtyReturned) {
            $data['qtyReturned'] = $qtyReturned;
        }
        if ($qtyApproved) {
            $data['qtyApproved'] = $qtyApproved;
        }
        return $data;
    }
}
