<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSOrderSync\Test\Unit\Core\SaaS;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Psr7\Response;
use Magento\Framework\Event\ManagerInterface as EventManager;
use Magento\SaaSOrderSync\Core\SaaS\AuditMiddleware;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Log\NullLogger;
use function PHPUnit\Framework\assertArrayHasKey;
use function PHPUnit\Framework\assertEquals;
use function PHPUnit\Framework\callback;
use function PHPUnit\Framework\equalTo;
use function PHPUnit\Framework\never;
use function PHPUnit\Framework\once;
use const false;

class AuditMiddlewareTest extends TestCase
{

    private EventManager&MockObject $eventManager;
    private AuditMiddleware $auditMiddleware;

    public function setUp(): void
    {
        $this->eventManager = $this->createMock(EventManager::class);
        $this->auditMiddleware = new AuditMiddleware($this->eventManager, new NullLogger());
    }

    public function test_shouldNotAudit_whenDisabled_byDefault()
    {
        $client = $this->createClient(auditOptions: ['enabled' => false]);
        $this->eventManager->expects(never())->method('dispatch');

        $response = $client->request('GET', '/');

        $this->assertEquals(200, $response->getStatusCode());
    }

    public function test_shouldNotAudit_whenDisabled_byRequest()
    {
        $client = $this->createClient(auditOptions: ['enabled' => true]);
        $this->eventManager->expects(never())->method('dispatch');

        $response = $client->request('GET', '/', ['audit' => ['enabled' => false]]);

        $this->assertEquals(200, $response->getStatusCode());
    }

    public function test_shouldNotAudit_whenEnabled_butMissingSender()
    {
        $client = $this->createClient(
            auditOptions: ['enabled' => true, 'destination' => 'd1', 'type' => 't1']
        );
        $this->eventManager->expects(never())->method('dispatch');

        $response = $client->request('GET', '/');

        $this->assertEquals(200, $response->getStatusCode());
    }

    public function test_shouldNotAudit_whenEnabled_butMissingDestination()
    {
        $client = $this->createClient(
            auditOptions: ['enabled' => true, 'sender' => 's1', 'type' => 't1']
        );
        $this->eventManager->expects(never())->method('dispatch');

        $response = $client->request('GET', '/');

        $this->assertEquals(200, $response->getStatusCode());
    }

    public function test_shouldNotAudit_whenEnabled_butMissingType()
    {
        $client = $this->createClient(
            auditOptions: ['enabled' => true, 'sender' => 's1', 'destination' => 'd1']
        );
        $this->eventManager->expects(never())->method('dispatch');

        $response = $client->request('GET', '/');

        $this->assertEquals(200, $response->getStatusCode());
    }

    public function test_shouldNotAudit_whenEnabled_butRequestFailed()
    {
        $this->expectException(ClientException::class);

        $client = $this->createClient(
            auditOptions: ['enabled' => true, 'sender' => 's1', 'destination' => 'd1', 'type' => 't1'],
            responseStatus: 400
        );

        $this->eventManager->expects(never())->method('dispatch');

        $client->request('GET', '/');
    }

    public function test_shouldAudit_whenEnabled()
    {
        $client = $this->createClient(
            auditOptions: ['enabled' => true, 'sender' => 's1', 'destination' => 'd1', 'type' => 't1'],
        );

        $this->eventManager
            ->expects(once())
            ->method('dispatch')
            ->with(equalTo('data_sent_outside'), callback(function (array $data = []): bool {
                assertEquals('s1', $data['sender']);
                assertEquals('d1', $data['destination']);
                assertEquals('t1', $data['type']);
                assertArrayHasKey('timestamp', $data);
                assertEquals('{"foo":"bar"}', $data['data']);
                return true;
            }));

        $client->request('POST', '/', ['json' => ['foo' => 'bar']]);
    }

    public function createClient(array $auditOptions = [], int $responseStatus = 200): Client
    {
        $mock = new MockHandler([
            new Response($responseStatus),
        ]);
        $handlerStack = HandlerStack::create($mock);
        $handlerStack->push($this->auditMiddleware->create($auditOptions));
        return new Client(['handler' => $handlerStack]);
    }

}
