<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSOrderSync\Core\SaaS;

use Magento\Framework\Event\ManagerInterface as EventManager;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Log\LoggerInterface;
use function array_diff;
use function array_keys;
use function implode;

class AuditMiddleware
{
    public function __construct(
        private readonly EventManager    $eventManager,
        private readonly LoggerInterface $logger,
    ) {
    }

    /**
     * Creates the auditing middleware.
     * Default options are merged with the request options under the 'audit' key where the latest have precedence.
     *
     * For instance,
     * ```php
     * // Middleware creation
     * $auditMiddleware->create([
     *   'enabled' => false,
     *   'sender' => 'bob',
     * ]);
     *
     * // Request
     * $client->request('POST', $url, [
     *   'audit' => ['enabled' => true, 'destination' => 'alice', 'type' => 'carol']
     * ]);
     *
     * // Will audit the following event
     * eventManager->dispatch('data_sent_outside', [
     *   'sender' => 'bob',
     *   'destination' => 'alice',
     *   'type' => 'carol',
     *   'timestamp' => 'unix timestamp',
     *   'data' => 'request payload',
     * ]);
     * ```
     *
     * @param array $defaultAuditOptions associative array with the default options:
     *   array[
     *     'enabled' => bool
     *     'sender' => string
     *     'destination' => string
     *     'type' => string
     *  ]
     * @return callable the auditing middleware which will only audit when the request is successful.
     */
    public function create(array $defaultAuditOptions = []): callable
    {
        $defaultOpts = array_merge([
            'enabled' => false,
        ], $defaultAuditOptions);

        return function (callable $handler) use ($defaultOpts) {
            return function (RequestInterface $request, array $options) use ($handler, $defaultOpts) {
                $auditOptions = array_merge($defaultOpts, $options['audit'] ?? []);

                if ($this->isNotEnabled($auditOptions) || $this->someOptionsAreMissing($request, $auditOptions)) {
                    return $handler($request, $options);
                }

                $promise = $handler($request, $options);
                return $promise->then(function (ResponseInterface $response) use ($request, $auditOptions) {
                    if ($this->isSuccessful($response)) {
                        $this->dispatchEvent($request, $auditOptions);
                    }

                    return $response;
                });
            };
        };
    }

    function isNotEnabled(array $auditOptions = []): bool
    {
        return !($auditOptions['enabled'] ?? false);
    }

    private function someOptionsAreMissing(RequestInterface $request, array $auditOptions = []): bool
    {
        $missingOpts = array_diff(['sender', 'destination', 'type'], array_keys($auditOptions));
        if (empty($missingOpts)) {
            return false;
        }

        $formattedMissingOpts = implode(",", $missingOpts);
        $this->logger->warning(
            "Skip audit for request [{$request->getMethod()}]{$request->getUri()}"
            ." since $formattedMissingOpts audit options are missing.");
        return true;
    }

    private function isSuccessful(ResponseInterface $response): bool
    {
        return $response->getStatusCode() >= 200 && $response->getStatusCode() < 300;
    }

    function dispatchEvent(RequestInterface $request, array $auditOptions): void
    {
        $body = $request->getBody();
        $body->rewind();

        $this->eventManager->dispatch(
            'data_sent_outside',
            [
                'sender' => $auditOptions['sender'],
                'destination' => $auditOptions['destination'],
                'timestamp' => gmdate('U'),
                'type' => $auditOptions['type'],
                'data' => $body->getContents(),
            ]
        );
    }
}
