<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
namespace Magento\SaaSOrderSync\Core\OrderSync\Bulk;

use Magento\AsynchronousOperations\Api\Data\OperationInterface;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\SaaSOrderSync\Api\OrderSync\Bulk\OperationData;
use Magento\SaaSOrderSync\Api\OrderSync\Bulk\OperationProcessorPool;
use Magento\SaaSOrderSync\Api\Result;
use Magento\SaaSOrderSync\Core\Bulk\BulkAdapter;
use Magento\SaaSOrderSync\Core\OrderSync\Config;
use Psr\Log\LoggerInterface;
use Symfony\Component\Stopwatch\Stopwatch;

class OperationConsumer
{
    private LoggerInterface $logger;
    private SerializerInterface $serializer;
    private BulkAdapter $bulkAdapter;
    private OperationProcessorPool $operationProcessorPool;
    private Config $config;

    public function __construct(
        LoggerInterface        $logger,
        SerializerInterface    $serializer,
        BulkAdapter            $bulkAdapter,
        OperationProcessorPool $operationProcessorPool,
        Config                 $config,
    ) {
        $this->logger = $logger;
        $this->serializer = $serializer;
        $this->bulkAdapter = $bulkAdapter;
        $this->operationProcessorPool = $operationProcessorPool;
        $this->config = $config;
    }

    public function consume(OperationInterface $operation): bool
    {
        $syncId = $operation->getBulkUuid();
        $operationId = $operation->getId();
        $ctx = ['syncId' => $syncId, 'operationId' => $operationId];
        $this->logger->debug('Order sync operation started.', $ctx);

        $stopwatch = new Stopwatch(true);
        $stopwatch->start("$syncId-$operationId");
        [$err, $data] = $this->process($operation);
        $processingMillis = $stopwatch->stop("$syncId-$operationId")->getDuration();

        if ($err) {
            [$level, $message] = match ($err['code']) {
                'NOT_IN_PROGRESS' => ['WARNING', 'Operation has failed because order sync is not in progress.'],
                default => ['CRITICAL', 'Unexpected error during order sync operation.']
            };

            $result = [
                'error' => $err,
                'processing_time_ms' => $processingMillis,
            ];
            $this->bulkAdapter->failOperation($operation, "$level - $message", $result);

            $ctx['error'] = $err;
            $this->logger->log($level, $message, $ctx);
            return false;
        }

        $result = [
            'responses' => $data,
            'processing_time_ms' => $processingMillis,
        ];
        $this->bulkAdapter->completeOperation($operation, $result);

        $this->logger->debug('Order sync operation finished.', $ctx);
        return true;
    }

    private function process(OperationInterface $operation): Result
    {
        return Result::wrap(function () use ($operation) {
            $ctx = ['syncId' => $operation->getBulkUuid(), 'operationId' => $operation->getId()];
            $operationData = new OperationData($this->serializer->unserialize($operation->getSerializedData()));
            $results = [];

            $operationProcessors = $this->operationProcessorPool->get();
            $allOrderIds = $operationData->getOrderIds();
            $syncRequestSize = $this->config->getOperationSyncRequestOrderCount();

            $someError = false;
            foreach (array_chunk($allOrderIds, $syncRequestSize) as $index => $orderIds) {
                $ctx['index'] = $index;
                foreach ($operationProcessors as $operationProcessor) {
                    $class = $operationProcessor::class;

                    $this->logger->debug("Start execution of operation processor $class.", $ctx);
                    $result = $operationProcessor->process($operation->getBulkUuid(), $orderIds);
                    $this->logger->debug("End execution of operation processor $class.", array_merge($ctx, ['result' => $result]));

                    [$err, $data] = $result;
                    $results[$index][$class] = $err ?: $data;
                    $someError = $err != null;
                }
            }

            if ($someError) {
                return Result::error('SOME_SYNC_REQUEST_FAILED', $results);
            }
            return $results;
        });
    }
}
