<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */

declare(strict_types=1);

namespace Magento\SaaSOrderSync\Test\Api\Core\OrderSync;

use DateTime;
use DateTimeInterface;
use Magento\Framework\Webapi\Rest\Request;
use Magento\Integration\Api\AdminTokenServiceInterface;
use Magento\TestFramework\Bootstrap as TestBootstrap;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\TestCase\WebapiAbstract;
use function PHPUnit\Framework\assertEquals;
use function PHPUnit\Framework\assertNotEmpty;
use function PHPUnit\Framework\assertNotNull;
use function PHPUnit\Framework\assertTrue;

/**
 * Tests a usual flow for sync calling real API.
 */
class OrderSyncManagerApiTest extends WebapiAbstract
{

    const ORDER_SYNC_PATH = '/V1/saasOrderSync/orderSync';

    const ORDER_SYNC_STATUS_PATH = '/V1/saasOrderSync/orderSync/:syncId';

    const ORDER_SYNC_CANCEL_PATH = '/V1/saasOrderSync/orderSync/:syncId/cancel';

    const SERVICE_NAME_SEARCH = 'saaSOrderSyncOrderSyncOrderSyncManagerV1';

    const SERVICE_VERSION = 'V1';

    /**
     *   Object manager to instantiate objects.
     *
     * @var \Magento\Framework\ObjectManagerInterface
     */
    private $objectManager;

    /**
     * @var \Magento\Integration\Api\AdminTokenServiceInterface
     */
    private $adminTokens;

    /**
     *  Test setup.
     *
     * @return void
     */
    protected function setUp(): void
    {
        $this->objectManager = Bootstrap::getObjectManager();
        $this->adminTokens   = $this->objectManager->get(AdminTokenServiceInterface::class);
        $this->createOrLoadAdminUser();
        $flagManager = $this->objectManager->get('\Magento\Framework\FlagManager');
        $flagManager->saveFlag('running_webapi_testmode', 'true');
    }

    protected function tearDown(): void
    {
        $flagManager = $this->objectManager->get('\Magento\Framework\FlagManager');
        $flagManager->deleteFlag('running_webapi_testmode');
    }

    /**
     * Do a flow test.
     *
     * @return void
     */
    public function test_doSyncFlow()
    {
        $this->_markTestAsRestOnly('SaaS Order Sync API is only available in REST.');

        $token = $this->login();

        $allSyncs = $this->getAllSyncs($token);
        assertNotNull($allSyncs);

        $syncId = $this->createSync($token);
        assertNotNull($syncId);
        $allSyncs = $this->getAllSyncs($token);
        assertNotNull($allSyncs);

        $cancelledSync = $this->cancelSync($token, $syncId);
        assertNotNull($cancelledSync);

        $status = $this->getStatusSync($token, $syncId);
        assertNotNull($status);
        assertNotEmpty($status['order_sync']['status']);

        try {
            $status = $this->cancelSync($token, $syncId);
            assertNotNull($status);
        } catch (\Exception $e) {
            assertNotNull($e);
            assertEquals(409, $e->getCode());
            assertTrue(str_contains($e->getMessage(), 'is not in progress'));
        }
    }

    /**
     * Get all current syncs.
     *
     * @param $token authorization token
     * @return string response with all syncs.
     */
    private function getAllSyncs($token): array
    {

        $serviceInfo = [
            'rest' => [
                'resourcePath' => self::ORDER_SYNC_PATH,
                'httpMethod'   => Request::HTTP_METHOD_GET,
                'Content-Type' => 'application/json',
                'token'        => $token,
            ],
        ];

        return $this->_webApiCall($serviceInfo, [], WebapiAbstract::ADAPTER_REST);
    }

    private function createSync($token): string
    {

        $createdFrom = (new DateTime('2022-01-01'))->format(DateTimeInterface::ATOM);
        $createdTo   = (new DateTime('2024-01-01'))->format(DateTimeInterface::ATOM);
        $clientCode  = 'unit-test';

        $serviceInfo = [
           'rest' => [
               'resourcePath' => self::ORDER_SYNC_PATH,
               'httpMethod'   => Request::HTTP_METHOD_POST,
               'Content-Type' => 'application/json',
               'token'        => $token,
           ],
        ];

        $arguments = [
            'client_code'  => $clientCode,
            'created_from' => $createdFrom,
            'created_to'   => $createdTo,

        ];

        $response = $this->_webApiCall($serviceInfo, $arguments, WebapiAbstract::ADAPTER_REST);
        return $response['order_sync']['sync_id'];
    }

    private function cancelSync($token, $syncId): array
    {
        $serviceInfo = [
            'rest' => [
                'resourcePath' => str_replace(':syncId', $syncId, self::ORDER_SYNC_CANCEL_PATH),
                'httpMethod'   => Request::HTTP_METHOD_PUT,
                'Content-Type' => 'application/json',
                'token'        => $token,
            ],
        ];

        return $this->_webApiCall($serviceInfo, [], WebapiAbstract::ADAPTER_REST);
    }

    private function getStatusSync($token, $syncId): array
    {
        $serviceInfo = [
            'rest' => [
                'resourcePath' => str_replace(':syncId', $syncId, self::ORDER_SYNC_STATUS_PATH),
                'httpMethod'   => Request::HTTP_METHOD_GET,
                'Content-Type' => 'application/json',
                'token'        => $token,
            ],
        ];

        return $this->_webApiCall($serviceInfo, [], WebapiAbstract::ADAPTER_REST);
    }

    /**
     * @return void
     */
    private function createOrLoadAdminUser(): void
    {
        $userFactory = $this->objectManager->get('\Magento\User\Model\UserFactory');
        $user = $userFactory->create();
        $user->loadByUsername(TestBootstrap::ADMIN_NAME);

        if ($user->getUserName() === null) {
            $user->setRoleId(1) // 1 is administrator unless something has been customized
            ->setFirstname('Admin')
                ->setLastname('Admin')
                ->setActive(true)
                ->setUserName(TestBootstrap::ADMIN_NAME)
                ->setPassword(TestBootStrap::ADMIN_PASSWORD)
                ->setEmail('admin1@test.com')
                ->save();
        } else {
            if ($user->getIsActive() === false) {
                $user->setRoleId(1)->setActive(true)->save();
            }
        }
    }

    private function login(): string
    {
        return $this->adminTokens->createAdminAccessToken(
            \Magento\TestFramework\Bootstrap::ADMIN_NAME,
            \Magento\TestFramework\Bootstrap::ADMIN_PASSWORD
        );
    }
}
