<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSOrderSync\Core\OrderSync;

use function __;
use DateTime;
use Exception;
use Magento\Framework\DataObject\IdentityGeneratorInterface;
use Magento\Framework\FlagManager;
use Magento\Framework\Webapi\Exception as WebapiException;
use Magento\SaaSOrderSync\Api\OrderSync\CancelOrderSyncResponseInterface;
use Magento\SaaSOrderSync\Api\OrderSync\CreateOrderSyncResponseInterface;
use Magento\SaaSOrderSync\Api\OrderSync\GetAllOrderSyncsResponseInterface;
use Magento\SaaSOrderSync\Api\OrderSync\GetOneOrderSyncResponseInterface;
use Magento\SaaSOrderSync\Api\OrderSync\OrderSyncManagerInterface;
use Magento\SaaSOrderSync\Core\CommerceDataExport\OrdersDataExporterAdapter;
use Magento\SaaSOrderSync\Core\Common\Time\ClockInterface;
use Magento\SaaSOrderSync\Core\OrderSync\Bulk\Manager as BulkManager;
use Magento\SaaSOrderSync\Core\SaaS\SaaSClientResolverInterface;
use Magento\SaaSOrderSync\Support\Logging;
use Psr\Log\LoggerInterface;
use Symfony\Component\Stopwatch\Stopwatch;
use Throwable;

class Manager implements OrderSyncManagerInterface
{

    public function __construct(
        private readonly IdentityGeneratorInterface  $identityGenerator,
        private readonly OrdersDataExporterAdapter   $ordersExporter,
        private readonly BulkManager                 $ordersBulkManager,
        private readonly SaaSClientResolverInterface $clientResolver,
        private readonly LoggerInterface             $logger,
        private readonly Lock                        $orderSyncLock,
        private readonly ClockInterface              $clock,
        private readonly FlagManager                 $flagManager,
    ) {
    }

    public function createOrderSync(
        string $client_code,
        string $created_from,
        string $created_to = 'now'
    ): CreateOrderSyncResponseInterface {

        if ($this->isRunningWebApiTestMode()) {
            return new CreateOrderSyncResponse($this->testSync());
        }

        $syncId = $this->identityGenerator->generateId();
        $ctx = [
            'syncId' => $syncId,
            'clientCode' => $client_code,
            'createdFrom' => $created_from,
            'createdTo' => $created_to
        ];

        $clientCode = $this->parseString($client_code)
            ?: $this->throwBadRequest('client_code can\'t be empty.');
        $createdFrom = $this->validateCreatedFrom($created_from);
        $createdTo = $this->validateCreatedTo($created_to);

        if (!$this->ordersBulkManager->isAMQPAvailable()) {
            throw new WebapiException(
                __(
                    'Can\'t start Order Sync due to missing AMQP connection. Request to the administrator of the'
                    . ' Adobe Commerce instance to configure Message Queue Framework accordingly.'
                    . ' See: https://experienceleague.adobe.com/docs/commerce-operations/configuration-guide/message-queues/message-queue-framework.html?lang=en'
                ),
                httpCode: WebapiException::HTTP_INTERNAL_ERROR,
            );
        }

        try {
            $initialCount = $this->ordersExporter->count($createdFrom, $createdTo);
            if ($initialCount == 0) {
                return $this->createCompletedSync($syncId, $clientCode, $createdFrom, $createdTo);
            }

            $orderSyncClient = $this->clientResolver->createOrderSyncClient();
            $orderSync = $orderSyncClient->createOrderSync(
                $syncId,
                $createdFrom,
                $createdTo,
                $initialCount,
                $clientCode
            );

            $stopwatch = new Stopwatch(true);
            $stopwatch->start($syncId);
            $this->ordersBulkManager->schedule($syncId, $createdTo, $createdFrom);
            $creationMillis = $stopwatch->stop($syncId)->getDuration();

            $this->logger->info("Created SaaS order sync, took {$creationMillis}ms.", $ctx);

            return new CreateOrderSyncResponse($this->map($orderSync));
        } catch (WebapiException $e) {
            throw $e;
        } catch (Throwable $e) {
            $this->logAndRethrow($e, "Unexpected exception on " . (__METHOD__) . ".", $ctx);
        }
    }

    public function getAllOrderSyncs(): GetAllOrderSyncsResponseInterface
    {
        if ($this->isRunningWebApiTestMode()) {
            return new GetAllOrderSyncsResponse($this->testSyncArray());
        }
        try {
            $orderSyncClient = $this->clientResolver->createOrderSyncClient();
            ['orderSyncs' => $orderSyncs] = $orderSyncClient->getAllOrderSyncs();

            // @formatter:off
            return new GetAllOrderSyncsResponse(array_map(
                fn (array $orderSync): OrderSync => $this->map($orderSync),
                $orderSyncs
            ));
            // @formatter:on
        } catch (Throwable $e) {
            $this->logAndRethrow($e, "Unexpected exception on " . (__METHOD__) . ".");
        }
    }

    public function getOneOrderSync($sync_id): GetOneOrderSyncResponseInterface
    {
        if ($this->isRunningWebApiTestMode()) {
            return new GetOneOrderSyncResponse($this->testSync());
        }
        try {
            $orderSyncClient = $this->clientResolver->createOrderSyncClient();
            $orderSync = $orderSyncClient->getOneOrderSync($sync_id);

            if ($orderSync == null) {
                throw new WebapiException(__("No order sync found for id '$sync_id'."));
            }

            return new GetOneOrderSyncResponse($this->map($orderSync));
        } catch (WebapiException $e) {
            throw $e;
        } catch (Throwable $e) {
            $this->logAndRethrow($e, "Unexpected exception on " . (__METHOD__) . ".", ['syncId' => $sync_id]);
        }
    }

    public function cancelOrderSync(string $sync_id): CancelOrderSyncResponseInterface
    {
        if ($this->isRunningWebApiTestMode()) {
            return new CancelOrderSyncResponse($this->testSync());
        }
        try {
            return $this->orderSyncLock->locking($sync_id, function ($syncId) {
                $orderSyncClient = $this->clientResolver->createOrderSyncClient();
                [$err, $data] = $orderSyncClient->cancelOrderSync($syncId);

                if ($err) {
                    match ($err['code']) {
                        'NOT_FOUND' => throw new WebapiException(__("No order sync found for id '$syncId'.")),
                        'NOT_IN_PROGRESS' => throw new WebapiException(__("Order sync '$syncId' is not in progress."), httpCode: 409),
                        default => $this->logAndThrow(
                            'Unexpected error while canceling order sync in SaaS.',
                            ['syncId' => $syncId, 'error' => $err]
                        )
                    };
                }

                $this->ordersBulkManager->cancel($syncId);

                $this->logger->info('SaaS order sync has been canceled.', ['syncId' => $syncId]);

                return new CancelOrderSyncResponse($this->map($data));
            });
        } catch (WebapiException $e) {
            throw $e;
        } catch (Throwable $e) {
            $this->logAndRethrow($e, "Unexpected exception on " . (__METHOD__) . ".", ['syncId' => $sync_id]);
        }
    }

    private function map(array $orderSync): OrderSync
    {
        return new OrderSync(
            $orderSync['syncId'],
            $orderSync['createdFrom'],
            $orderSync['createdTo'],
            $orderSync['clientCode'],
            $orderSync['initialCount'],
            $orderSync['processedCount'],
            $orderSync['errorCount'],
            $orderSync['status']
        );
    }

    private function validateCreatedFrom(string $createdFrom): DateTime
    {
        $parsedCreatedFrom = $this->parseDateTime($createdFrom)
            ?: $this->throwBadRequest('created_from can\'t be parsed in ISO8601 format.');

        if ($this->clock->now() < $parsedCreatedFrom) {
            $this->throwBadRequest('created_from should be in the past.');
        }

        return $parsedCreatedFrom;
    }

    private function validateCreatedTo(string $createdTo): DateTime
    {
        if ($createdTo === 'now') {
            return new DateTime('now');
        }

        return $this->parseDateTime($createdTo)
            ?: $this->throwBadRequest('created_to can\'t be parsed in ISO8601 format.');
    }

    public function parseDateTime(string $dateTime): DateTime|bool
    {
        try {
            return new DateTime($dateTime);
        } catch (Exception) {
            return false;
        }
    }

    private function parseString(string $string): string|bool
    {
        return strlen(trim($string)) === 0 ? false : $string;
    }

    private function createCompletedSync(
        string   $syncId,
        string   $clientCode,
        DateTime $createdFrom,
        DateTime $createdTo
    ): CreateOrderSyncResponseInterface {
        $orderSyncClient = $this->clientResolver->createOrderSyncClient();
        $orderSyncClient->createOrderSync($syncId, $createdFrom, $createdTo, 0, $clientCode);
        [$err, $data] = $orderSyncClient->completeOrderSync($syncId);
        if ($err) {
            $this->logAndThrow('Failed to complete order sync.', ['syncId' => $syncId]);
        }

        return new CreateOrderSyncResponse($this->map($data));
    }

    private function logAndThrow(string $msg, array $ctx = [])
    {
        $e = new WebapiException(
            __($msg),
            httpCode: WebapiException::HTTP_INTERNAL_ERROR
        );
        $this->logAndRethrow($e, $msg, $ctx);
    }

    private function logAndRethrow(Throwable $e, string $msg, array $ctx = [])
    {
        $ctx['exception'] = Logging::formatException($e);
        $this->logger->critical($msg, $ctx);
        throw $e;
    }

    private function throwBadRequest($msg)
    {
        throw new WebapiException(
            __($msg),
            httpCode: WebapiException::HTTP_BAD_REQUEST,
        );
    }

    private function isRunningWebApiTestMode(): bool
    {
        return $this->flagManager->getFlagData('running_webapi_testmode') === 'true';
    }

    /**
     * @return array
     */
    public function testSyncArray(): array
    {
        return [
            0 => $this->testSync(),
        ];
    }

    /**
     * @return OrderSync
     */
    public function testSync(): OrderSync
    {
        return new OrderSync(
            'test_sync_id',
            '2022-01-01T00:00:00+00:00',
            '2024-01-01T00:00:00+00:00',
            'test_client_code',
            0,
            0,
            0,
            'completed'
        );
    }
}
