<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2024 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);
namespace Magento\SaaSOrderSync\Console\Command;

use Magento\AsynchronousOperations\Api\Data\OperationInterface;
use Magento\Framework\Bulk\BulkSummaryInterface;
use Magento\SaaSOrderSync\Api\OrderSync\OrderSyncInterface;

class ResponseFormatter
{

    /**
     * @param OrderSyncInterface $sync
     * @return false|string
     */
    public function format(OrderSyncInterface $sync, OrderSyncBulkInfo $bulkInfo = null): string|false
    {
        $data = [
            "syncId" => $sync->getSyncId(),
            "createdFrom" => $sync->getCreatedFrom(),
            "createdTo" => $sync->getCreatedTo(),
            "clientCode" => $sync->getClientCode(),
            "status" => $sync->getStatus(),
            "initialCount" => $sync->getInitialCount(),
            "processedCount" => $sync->getProcessedCount(),
            "errorCount" => $sync->getErrorCount(),
        ];

        if ($bulkInfo != null) {
            $data['bulk'] = $this->formatBulkData($bulkInfo);
        }

        return json_encode($data, JSON_PRETTY_PRINT);
    }

    private function formatBulkStatus(int $bulkStatusId): string
    {
        $status = match ($bulkStatusId) {
            BulkSummaryInterface::NOT_STARTED => 'NOT_STARTED',
            BulkSummaryInterface::IN_PROGRESS => 'IN_PROGRESS',
            BulkSummaryInterface::FINISHED_SUCCESSFULLY => 'FINISHED_SUCCESSFULLY',
            BulkSummaryInterface::FINISHED_WITH_FAILURE => 'FINISHED_WITH_FAILURE',
            default => 'UNKNOWN'
        };

        return "$status ($bulkStatusId)";
    }

    private function formatOperationStatus(int $operationStatusId): string
    {

        $status = match ($operationStatusId) {
            OperationInterface::STATUS_TYPE_COMPLETE => 'COMPLETE',
            OperationInterface::STATUS_TYPE_RETRIABLY_FAILED => 'RETRIABLY_FAILED',
            OperationInterface::STATUS_TYPE_NOT_RETRIABLY_FAILED => 'NOT_RETRIABLY_FAILED',
            OperationInterface::STATUS_TYPE_OPEN => 'OPEN',
            OperationInterface::STATUS_TYPE_REJECTED => 'REJECTED',
            default => 'UNKNOWN'
        };

        return "$status ($operationStatusId)";
    }

    /**
     * @param OrderSyncBulkInfo $bulkinfo bulk info type.
     * @return array
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function formatBulkData(OrderSyncBulkInfo $bulkinfo): array
    {
        $bulkDetail = $bulkinfo->getBulkDetail();
        $bulkData = [
            'bulkId'         => $bulkDetail->getBulkId(),
            'bulkStatus'     => $this->formatBulkStatus($bulkinfo->getBulkStatusId()),
            'operationCount' => $bulkDetail->getOperationCount(),
            'operations'     => array_map(
                fn (OperationInterface $operation): array => [
                    'operationId' => $operation->getId(),
                    'status' => $this->formatOperationStatus((int)$operation->getStatus()),
                    'errorCode' => $operation->getErrorCode(),
                    'resultMessage' => $operation->getResultMessage(),
                    'resultSerializedData' => $operation->getResultSerializedData() ? json_decode($operation->getResultSerializedData()) : null
                ],
                array_values($bulkDetail->getOperationsList())
            )
        ];

        return $bulkData;
    }
}
