<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSOrderSync\Console\Command;

use Magento\AsynchronousOperations\Api\BulkStatusInterface;
use Magento\Framework\Console\Cli;
use Magento\Framework\Webapi\Exception as WebapiException;
use Magento\SaaSOrderSync\Api\OrderSync\OrderSyncManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class GetOrderSyncStatus extends Command
{
    private const COMMAND_NAME = 'saas:order-sync:status';

    private OrderSyncManagerInterface $orderSyncManager;
    private ResponseFormatter $responseFormatter;
    private BulkStatusInterface $bulkStatus;

    public function __construct(OrderSyncManagerInterface $orderSyncManager, ResponseFormatter $responseFormatter, BulkStatusInterface $bulkStatus)
    {
        $this->orderSyncManager = $orderSyncManager;
        $this->responseFormatter = $responseFormatter;
        $this->bulkStatus = $bulkStatus;
        parent::__construct();
    }

    /**
     * @param \Magento\SaaSOrderSync\Api\OrderSync\OrderSyncInterface $sync
     * @return OrderSyncBulkInfo
     */
    public function getOrderSyncBulkInfo(\Magento\SaaSOrderSync\Api\OrderSync\OrderSyncInterface $sync): OrderSyncBulkInfo
    {
        $bulkDetail = $this->bulkStatus->getBulkDetailedStatus($sync->getSyncId());
        $bulkStatusId = $this->bulkStatus->getBulkStatus($bulkDetail->getBulkId());
        $bulkInfo = new OrderSyncBulkInfo($bulkDetail, $bulkStatusId);
        return $bulkInfo;
    }

    protected function configure(): void
    {
        $this
            ->setName(self::COMMAND_NAME)
            ->setDescription('Get the status of a SaaS order synchronization.')
            ->addArgument(
                'syncId',
                InputArgument::REQUIRED,
                'Sync id'
            )
            ->addOption('show-bulk-detail', 'b', InputOption::VALUE_NONE, "Show detail of the bulk.");
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $syncId = $input->getArgument('syncId');

        try {
            $response = $this->orderSyncManager->getOneOrderSync($syncId);
            $sync     = $response->getOrderSync();

            $showBulkDetail = $input->getOption('show-bulk-detail');
            if ($showBulkDetail) {
                $bulkInfo = $this->getOrderSyncBulkInfo($sync);
                $pretty   = $this->responseFormatter->format($sync, $bulkInfo);
            } else {
                $pretty = $this->responseFormatter->format($sync);
            }

            $output->writeln("<info>$pretty</info>");
            return Cli::RETURN_SUCCESS;
        } catch (WebapiException $e) {
            $output->writeln("<error>{$e->getHttpCode()}</error>");
            return Cli::RETURN_FAILURE;
        }
    }
}
