<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSOrderSync\Test\Unit\Core\OrderSync;

use DateTime;
use DateTimeInterface;
use Exception;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Framework\Webapi\Exception as WebapiException;
use Magento\SaaSOrderSync\Api\OrderSync\Bulk\OperationData;
use Magento\SaaSOrderSync\Api\Result;
use Magento\SaaSOrderSync\Core\Bulk\BulkAdapter;
use Magento\SaaSOrderSync\Core\CommerceDataExport\OrdersDataExporterAdapter;
use Magento\SaaSOrderSync\Core\OrderSync\Bulk\Manager as OrderSyncBulkManager;
use Magento\SaaSOrderSync\Core\OrderSync\LocalOrderSyncRepository;
use Magento\SaaSOrderSync\Core\OrderSync\Lock;
use Magento\SaaSOrderSync\Core\OrderSync\Manager as OrderSyncManager;
use Magento\SaaSOrderSync\Core\OrderSync\SaaSClient;
use Magento\SaaSOrderSync\Test\Unit\Core\Common\Generators;
use Magento\SaaSOrderSync\Test\Unit\Core\Common\Time\ClockMock;
use Magento\SaaSOrderSync\Test\Unit\Core\SaaS\SaaSClientMockResolver;
use PHPUnit\Framework\Constraint\Constraint;
use PHPUnit\Framework\Constraint\LogicalAnd;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Log\NullLogger;
use function PHPUnit\Framework\arrayHasKey;
use function PHPUnit\Framework\assertEquals;
use function PHPUnit\Framework\assertNotNull;
use function PHPUnit\Framework\assertThat;
use function PHPUnit\Framework\callback;
use function PHPUnit\Framework\equalTo;
use function PHPUnit\Framework\exactly;
use function PHPUnit\Framework\isInstanceOf;
use function PHPUnit\Framework\logicalNot;
use function PHPUnit\Framework\never;
use function PHPUnit\Framework\once;

class OrderSyncManagerTest extends TestCase
{
    private SaaSClient&MockObject $orderSyncClient;
    private OrdersDataExporterAdapter&MockObject $exporterAdapterMock;
    private BulkAdapter&MockObject $bulkAdapterMock;
    private LocalOrderSyncRepository&MockObject $localOrderSyncRepositoryMock;
    private ClockMock $clockMock;
    private FixedIdentityGenerator $identityGenerator;

    private OrderSyncManager $orderSyncManager;

    public function setUp(): void
    {
        $logger = new NullLogger();

        $this->identityGenerator = new FixedIdentityGenerator();

        $this->exporterAdapterMock = $this->createMock(OrdersDataExporterAdapter::class);
        $this->bulkAdapterMock = $this->createMock(BulkAdapter::class);
        $this->localOrderSyncRepositoryMock = $this->createMock(LocalOrderSyncRepository::class);

        $clientResolver = new SaaSClientMockResolver($this);
        $this->orderSyncClient = $clientResolver->createOrderSyncClient();

        $this->clockMock = new ClockMock();

        $this->orderSyncManager = new OrderSyncManager(
            $this->identityGenerator,
            $this->exporterAdapterMock,
            new OrderSyncBulkManager(
                $this->exporterAdapterMock,
                new Json(),
                $this->localOrderSyncRepositoryMock,
                $logger,
                $this->bulkAdapterMock
            ),
            $clientResolver,
            $logger,
            new Lock($logger, new NoLock()),
            $this->clockMock
        );

        $this->bulkAdapterMock->method('isAMQPAvailable')
            ->withAnyParameters()
            ->willReturn(true);
    }

    public function test_shouldCreateSyncUntilCurrentTime_whenThereAreOrders()
    {
        $syncId = $this->identityGenerator->getId();
        $createdFrom = (new DateTime('2022-01-01'))->format(DateTimeInterface::ATOM);
        $createdFromDateTime = new DateTime($createdFrom);
        $createdToDateTime = (new DateTime('2024-01-01'));
        $createdTo = $createdToDateTime->format(DateTimeInterface::ATOM);

        $clientCode = 'unit-test';

        $this->exporterAdapterMock->expects(once())
            ->method('count')
            ->with($createdFromDateTime, $createdToDateTime)
            ->willReturn(5);

        $this->exporterAdapterMock->expects(once())
            ->method('findOrderIds')
            ->with($createdFromDateTime, $createdToDateTime)
            ->willReturn(Generators::fixed([1, 2], [3, 4], [5]));

        $this->orderSyncClient->expects(once())
            ->method('createOrderSync')
            ->with($syncId, $createdFromDateTime, $createdToDateTime, 5, $clientCode)
            ->willReturnCallback($this->orderSyncResponse());

        $bulkArg = $this->and(
            $this->arrayHasKeyAndValue('uuid', $syncId),
            arrayHasKey('description')
        );
        $this->bulkAdapterMock->expects(exactly(3))
            ->method('scheduleOperation')
            ->withConsecutive(
                [$bulkArg, 1, new OperationData(['orderIds' => [1, 2]])],
                [$bulkArg, 2, new OperationData(['orderIds' => [3, 4]])],
                [$bulkArg, 3, new OperationData(['orderIds' => [5]])],
            )
            ->willReturn(Result::data());

        $response = $this->orderSyncManager->createOrderSync($clientCode, $createdFrom, $createdTo);

        assertNotNull($response->getOrderSync()->getSyncId());
    }

    public function test_shouldCreateAndCompleteSync_whenTimeRangeIsAlreadySync()
    {
        $syncId = $this->identityGenerator->getId();
        $createdFrom = (new DateTime('2022-01-01'))->format(DateTimeInterface::ATOM);
        $createdFromDateTime = new DateTime($createdFrom);
        $clientCode = 'unit-test';

        $this->clockMock->setNow('2023-01-01');
        $createdToDateTime = $this->clockMock->now();
        $createdTo = $createdToDateTime->format(DateTimeInterface::ATOM);

        $this->orderSyncClient->expects(once())
            ->method('createOrderSync')
            ->with($syncId, $createdFromDateTime, $createdToDateTime, 0, $clientCode)
            ->willReturnCallback($this->orderSyncResponse());

        $this->orderSyncClient->expects(once())
            ->method('completeOrderSync')
            ->with($syncId)
            ->willReturn(Result::data([
                'syncId' => $syncId,
                'createdFrom' => $createdFrom,
                'createdTo' => $createdTo,
                'initialCount' => 0,
                'clientCode' => $clientCode,
                'processedCount' => 0,
                'errorCount' => 0,
                'status' => 'COMPLETED',
            ]));

        $response = $this->orderSyncManager->createOrderSync($clientCode, $createdFrom, $createdTo);

        assertNotNull($response->getOrderSync()->getSyncId());
        assertEquals($createdFrom, $response->getOrderSync()->getCreatedFrom());
        assertEquals($createdTo, $response->getOrderSync()->getCreatedTo());
        assertEquals('COMPLETED', $response->getOrderSync()->getStatus());
        assertEquals(0, $response->getOrderSync()->getInitialCount());
        assertEquals(0, $response->getOrderSync()->getProcessedCount());
        assertEquals(0, $response->getOrderSync()->getErrorCount());
    }

    public function test_shouldCreateAndCompleteSync_whenNoOrdersInTheTimeRange()
    {
        $syncId = $this->identityGenerator->getId();
        $createdFrom = (new DateTime('2022-01-01'))->format(DateTimeInterface::ATOM);
        $createdFromDateTime = new DateTime($createdFrom);
        $createdToDateTime = (new DateTime('2024-01-01'));
        $createdTo = $createdToDateTime->format(DateTimeInterface::ATOM);
        $clientCode = 'unit-test';

        $this->exporterAdapterMock->expects(once())
            ->method('count')
            ->with($createdFromDateTime, $createdToDateTime)
            ->willReturn(0);

        $this->exporterAdapterMock->expects(never())
            ->method(logicalNot(equalTo('count')));

        $this->orderSyncClient->expects(once())
            ->method('createOrderSync')
            ->with($syncId, $createdFromDateTime, $createdToDateTime, 0, $clientCode)
            ->willReturnCallback($this->orderSyncResponse());

        $this->orderSyncClient->expects(once())
            ->method('completeOrderSync')
            ->with($syncId)
            ->willReturn(Result::data([
                'syncId' => $syncId,
                'createdFrom' => $createdFrom,
                'createdTo' => $createdTo,
                'initialCount' => 0,
                'clientCode' => $clientCode,
                'processedCount' => 0,
                'errorCount' => 0,
                'status' => 'COMPLETED',
            ]));

        $response = $this->orderSyncManager->createOrderSync($clientCode, $createdFrom, $createdTo);

        assertNotNull($response->getOrderSync()->getSyncId());
        assertEquals($createdFrom, $response->getOrderSync()->getCreatedFrom());
        assertEquals($createdTo, $response->getOrderSync()->getCreatedTo());
        assertEquals('COMPLETED', $response->getOrderSync()->getStatus());
        assertEquals(0, $response->getOrderSync()->getInitialCount());
        assertEquals(0, $response->getOrderSync()->getProcessedCount());
        assertEquals(0, $response->getOrderSync()->getErrorCount());
    }

    public function test_shouldFailWithBadRequest_whenCreatedFromIsNotValid()
    {
        $createdFrom = 'not-valid';
        $clientCode = 'unit-test';

        $createdToDateTime = (new DateTime('2024-01-01'));
        $createdTo = $createdToDateTime->format(DateTimeInterface::ATOM);

        $this->assertThrowsWebapiException(
            fn () => $this->orderSyncManager->createOrderSync($clientCode, $createdFrom, $createdTo),
            WebapiException::HTTP_BAD_REQUEST
        );
    }

    public function test_shouldFailWithBadRequest_whenCreatedToIsNotValid()
    {
        $createdFromDateTime = (new DateTime('2024-01-01'));
        $createdFrom = $createdFromDateTime->format(DateTimeInterface::ATOM);
        $createdTo = 'not-valid';
        $clientCode = 'unit-test';

        $this->assertThrowsWebapiException(
            fn () => $this->orderSyncManager->createOrderSync($clientCode, $createdFrom, $createdTo),
            WebapiException::HTTP_BAD_REQUEST
        );
    }

    public function test_shouldFailWithBadRequest_whenCreatedFromIsNotInThePast()
    {
        $createdFrom = '2026-01-02';
        $createdToDateTime = (new DateTime('2024-01-01'));
        $createdTo = $createdToDateTime->format(DateTimeInterface::ATOM);
        $clientCode = 'unit-test';

        $this->assertThrowsWebapiException(
            fn () => $this->orderSyncManager->createOrderSync($clientCode, $createdFrom, $createdTo),
            WebapiException::HTTP_BAD_REQUEST
        );
    }

    public function test_shouldFailWithBadRequest_whenCreatedToIsNotHigherThanCreatedFrom()
    {
        $createdFromDateTime = (new DateTime('2026-01-01'));
        $createdFrom = $createdFromDateTime->format(DateTimeInterface::ATOM);
        $createdToDateTime = (new DateTime('2024-01-01'));
        $createdTo = $createdToDateTime->format(DateTimeInterface::ATOM);
        $clientCode = 'unit-test';

        $this->assertThrowsWebapiException(
            fn () => $this->orderSyncManager->createOrderSync($clientCode, $createdFrom, $createdTo),
            WebapiException::HTTP_BAD_REQUEST
        );
    }

    public function test_shouldFailWithBadRequest_whenClientIsNotValid()
    {
        $createdFromDateTime = (new DateTime('2022-01-01'));
        $createdFrom = $createdFromDateTime->format(DateTimeInterface::ATOM);
        $createdToDateTime = (new DateTime('2024-01-01'));
        $createdTo = $createdToDateTime->format(DateTimeInterface::ATOM);
        $clientCode = '';

        $this->assertThrowsWebapiException(
            fn () => $this->orderSyncManager->createOrderSync($clientCode, $createdFrom, $createdTo),
            WebapiException::HTTP_BAD_REQUEST
        );
    }

    private function arrayHasKeyAndValue(string $key, mixed $value): Constraint
    {
        return callback(fn (array $subject): bool => array_key_exists($key, $subject) && $subject[$key] == $value);
    }

    private function and(Constraint ...$constraints): Constraint
    {
        return LogicalAnd::fromConstraints(...$constraints);
    }

    private function assertThrowsWebapiException(callable $callable, int $httpStatusCode)
    {
        $this->assertThrowsException(
            $callable,
            WebapiException::class,
            fn (WebapiException $e) => assertThat($e->getHttpCode(), equalTo($httpStatusCode))
        );
    }

    private function assertThrowsException(callable $callable, string $exception, callable $exceptionCallable = null): void
    {
        try {
            $callable();
        } catch (Exception $e) {
            assertThat($e, isInstanceOf($exception));
            if ($exceptionCallable != null) {
                $exceptionCallable($e);
            }
        }
    }

    private function orderSyncResponse(string $status = 'IN_PROGRESS'): callable
    {
        return fn (string $syncId, DateTime $createdFrom, DateTime $createdTo, int $initialCount, string $clientCode): array => [
            'syncId' => $syncId,
            'createdFrom' => $createdFrom->format(DateTimeInterface::ATOM),
            'createdTo' => $createdTo->format(DateTimeInterface::ATOM),
            'initialCount' => $initialCount,
            'clientCode' => $clientCode,
            'processedCount' => 0,
            'errorCount' => 0,
            'status' => $status,
        ];
    }
}
