<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSOrderSync\Console\Command;

use Magento\Framework\Console\Cli;
use Magento\Framework\Webapi\Exception as WebapiException;
use Magento\SaaSOrderSync\Api\OrderSync\OrderSyncManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class StartOrderSync extends Command
{
    private const COMMAND_NAME = 'saas:order-sync:start';

    private OrderSyncManagerInterface $orderSyncManager;

    public function __construct(OrderSyncManagerInterface $orderSyncManager)
    {
        $this->orderSyncManager = $orderSyncManager;
        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->setName(self::COMMAND_NAME)
            ->setDescription('Synchronize orders with the SaaS from a certain point in time.')
            ->addArgument(
                'from',
                InputArgument::REQUIRED,
                'From date time in ISO-8601 format.'
            )
            ->addOption(
                'to',
                't',
                InputOption::VALUE_OPTIONAL,
                'To date time in ISO-8601 format.',
                'now'

            )
            ->addOption(
                'client-code',
                'c',
                InputOption::VALUE_OPTIONAL,
                'Client code will be persisted and emitted in the SaaS notifications for tracking purposes.',
                'cli',
            );
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $createdFrom = $input->getArgument('from');
        $createdTo = $input->getOption('to');
        $clientCode = $input->getOption('client-code');

        try {
            $response = $this->orderSyncManager->createOrderSync($clientCode, $createdFrom, $createdTo);
            $sync = $response->getOrderSync();

            $data = [
                "syncId" => $sync->getSyncId(),
                "createdFrom" => $sync->getCreatedFrom(),
                "createdTo" => $sync->getCreatedTo(),
                "clientCode" => $sync->getClientCode(),
                "status" => $sync->getStatus(),
                "initialCount" => $sync->getInitialCount(),
                "processedCount" => $sync->getProcessedCount(),
                "errorCount" => $sync->getErrorCount(),
            ];

            $pretty = json_encode($data, JSON_PRETTY_PRINT);
            $output->writeln("<info>$pretty</info>");

            return Cli::RETURN_SUCCESS;
        } catch (WebapiException $e) {
            $output->writeln("<error>{$e->getMessage()}</error>");
            return Cli::RETURN_FAILURE;
        }
    }
}
