<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSOrderSync\Test\Unit\Core\Merchant;

use Magento\SaaSOrderSync\Core\Merchant\MerchantManager;
use Magento\SaaSOrderSync\Core\Merchant\MerchantSaaSClient;
use Magento\SaaSOrderSync\Test\Unit\Core\SaaS\SaaSClientMockResolver;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Log\NullLogger;
use function PHPUnit\Framework\assertEquals;
use function PHPUnit\Framework\assertNotNull;
use function PHPUnit\Framework\once;

class MerchantManagerTest extends TestCase
{
    private MockObject&MerchantSaaSClient $merchantClient;

    private MerchantManager $merchantManager;

    public function setUp(): void
    {
        $logger = new NullLogger();
        $clientResolver = new SaaSClientMockResolver($this);
        $this->merchantClient = $clientResolver->createMerchantClient();
        $this->merchantManager = new MerchantManager(
            $clientResolver,
            $logger
        );
    }

    public function test_shouldReturnEmptySummary_whenThereAreNoOrdersInSaaS()
    {
        $emptyMerchantResult= [ 'orders' => [
            'totalCount' => 0,
            'syncErrorsCount' => 0]
        ];

        $this->merchantClient->expects(once())
            ->method('getDataSummary')
            ->willReturn($emptyMerchantResult);

        $response = $this->merchantManager->getMerchantDataSummary();

        assertNotNull($response);
        assertEquals(0, $response->getOrders()->getTotalCount());
        assertEquals(0, $response->getOrders()->getSyncErrorsCount());
        assertEquals("", $response->getOrders()->getCreatedFrom());
        assertEquals("", $response->getOrders()->getCreatedTo());

    }

    public function test_shouldReturnEmptySummary_whenThereAreNoOrdersNorInfoInSaaS()
    {

        $this->merchantClient->expects(once())
            ->method('getDataSummary')
            ->willReturn(['orders' => []]);

        $response = $this->merchantManager->getMerchantDataSummary();

        assertNotNull($response);
        assertEquals(0, $response->getOrders()->getTotalCount());
        assertEquals(0, $response->getOrders()->getSyncErrorsCount());
        assertEquals("", $response->getOrders()->getCreatedFrom());
        assertEquals("", $response->getOrders()->getCreatedTo());

    }

    public function test_shouldReturnSummary_whenThereAreOrdersInSaaS()
    {

        $dataIsSaaS= [ 'orders' => [
            'totalCount' => 1000,
            'syncErrorsCount' => 20,
            'createdFrom' => '2022-01-01T15:00Z',
            'createdTo' => '2023-01-01T15:00Z'
        ]];

        $this->merchantClient->expects(once())
            ->method('getDataSummary')
            ->willReturn($dataIsSaaS);

        $response = $this->merchantManager->getMerchantDataSummary();

        assertNotNull($response);
        assertEquals(1000, $response->getOrders()->getTotalCount());
        assertEquals(20, $response->getOrders()->getSyncErrorsCount());
        assertEquals("2022-01-01T15:00Z", $response->getOrders()->getCreatedFrom());
        assertEquals("2023-01-01T15:00Z", $response->getOrders()->getCreatedTo());
    }

}
