<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
namespace Magento\SaaSOrderSync\Core\OrderSync\Bulk;

use Magento\AsynchronousOperations\Api\Data\OperationInterface;
use Magento\SaaSOrderSync\Core\Bulk\BulkAdapter;
use Magento\SaaSOrderSync\Core\OrderSync\LocalOrderSyncRepository;
use Magento\SaaSOrderSync\Core\OrderSync\Lock;
use Magento\SaaSOrderSync\Core\SaaS\SaaSClientResolverInterface;
use Psr\Log\LoggerInterface;
use Throwable;

class OperationCompletionPlugin
{
    private LoggerInterface $logger;
    private Lock $orderSyncLock;
    private BulkAdapter $bulkAdapter;
    private SaaSClientResolverInterface $clientResolver;
    private LocalOrderSyncRepository $localOrderSyncRepository;

    public function __construct(
        LoggerInterface             $logger,
        Lock                        $lock,
        BulkAdapter                 $bulkStatus,
        SaaSClientResolverInterface $clientResolver,
        LocalOrderSyncRepository    $localOrderSyncRepository,
    ) {
        $this->logger = $logger;
        $this->orderSyncLock = $lock;
        $this->bulkAdapter = $bulkStatus;
        $this->clientResolver = $clientResolver;
        $this->localOrderSyncRepository = $localOrderSyncRepository;
    }

    /**
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function afterConsume(OperationConsumer $_, bool $result, OperationInterface $operation): bool
    {
        $syncId = $operation->getBulkUuid();
        $operationId = $operation->getId();

        $ctx = ['syncId' => $syncId, 'operationId' => $operationId];
        try {
            $this->orderSyncLock->locking($syncId, fn () => $this->complete($syncId, $operationId));
        } catch (Throwable $e) {
            $ctx['exception'] = $e;
            $this->logger->critical('Unexpected error in order sync complete plugin.', $ctx);
        }
        return $result;
    }

    public function complete(string $syncId, string $operationId): void
    {
        $ctx = ["syncId" => $syncId, "operationId" => $operationId];

        $localOrderSync = $this->localOrderSyncRepository->find($syncId);
        if ($localOrderSync && $localOrderSync->isCanceled()) {
            $this->logger->warning('Skip completion since order sync is canceled.', $ctx);
            return;
        }

        if (!$this->bulkAdapter->allOperationsCompleted($syncId)) {
            $this->logger->debug('Order sync is not completed.', $ctx);
            return;
        }

        $syncClient = $this->clientResolver->createOrderSyncClient();
        [$err, $data] = $syncClient->completeOrderSync($syncId);
        if ($err) {
            $errorCode = $err['code'];
            $level = match ($errorCode) {
                'NOT_FOUND', 'NOT_IN_PROGRESS' => 'WARNING',
                default => 'CRITICAL'
            };
            $ctx['error'] = $err;
            $this->logger->log($level, 'Unexpected error when completing order sync', $ctx);
            return;
        }

        $this->logger->info('Completed order sync.', $data);
    }
}
