<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
namespace Magento\SaaSOrderSync\Core\OrderSync\Bulk;

use DateTime;
use DateTimeInterface;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\SaaSOrderSync\Api\OrderSync\Bulk\OperationData;
use Magento\SaaSOrderSync\Core\Bulk\BulkAdapter;
use Magento\SaaSOrderSync\Core\CommerceDataExport\OrdersDataExporterAdapter;
use Magento\SaaSOrderSync\Core\OrderSync\LocalOrderSyncRepository;
use Psr\Log\LoggerInterface;

class Manager
{
    private OrdersDataExporterAdapter $ordersExporterAdapter;
    private SerializerInterface $serializer;
    private LocalOrderSyncRepository $localOrderSyncRepository;

    private LoggerInterface $logger;
    private BulkAdapter $bulkAdapter;

    public function __construct(
        OrdersDataExporterAdapter $ordersExporterAdapter,
        SerializerInterface       $serializer,
        LocalOrderSyncRepository  $localOrderSyncRepository,
        LoggerInterface           $logger,
        BulkAdapter               $bulkAdapter,
    ) {
        $this->ordersExporterAdapter = $ordersExporterAdapter;
        $this->serializer = $serializer;
        $this->localOrderSyncRepository = $localOrderSyncRepository;
        $this->logger = $logger;
        $this->bulkAdapter = $bulkAdapter;
    }

    public function schedule(string $syncId, DateTime $createdTo, DateTime $createdFrom): string
    {
        $bulk = [
            'uuid' => $syncId,
            'description' => 'SaaS order sync'
                . " from {$createdFrom->format(DateTimeInterface::ATOM)}"
                . " to {$createdTo->format(DateTimeInterface::ATOM)}"
        ];

        $operationsCount = 0;
        foreach ($this->ordersExporterAdapter->findOrderIds($createdFrom, $createdTo) as $orderIds) {
            $operationId = ++$operationsCount;
            $ctx = ['syncId' => $syncId, 'operationId' => $operationId];

            $data = new OperationData();
            $data->setOrderIds($orderIds);

            [$err] = $this->bulkAdapter->scheduleOperation($bulk, $operationId, $data);
            if ($err) {
                $ctx['error'] = $err;
                $orderIdsList = $this->serializer->serialize($orderIds);
                $this->logger->error("Failed to create operation for order sync bulk. The following ids won't be exported: $orderIdsList", $ctx);
            } else {
                $this->logger->debug("Added operation to order sync bulk.", $ctx);
            }
        }

        $this->logger->debug("Scheduled order sync bulk with $operationsCount operations.", ['syncId' => $syncId]);
        return $syncId;
    }

    public function cancel(string $syncId): void
    {
        $this->localOrderSyncRepository->upsert($syncId, new DateTime());
    }

    public function isAMQPAvailable(): bool
    {
        return $this->bulkAdapter->isAMQPAvailable();
    }
}
