<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
namespace Magento\SaaSOrderSync\Core\Bulk;

use LogicException;
use Magento\AsynchronousOperations\Api\BulkStatusInterface;
use Magento\AsynchronousOperations\Api\Data\OperationInterface;
use Magento\AsynchronousOperations\Api\Data\OperationInterfaceFactory;
use Magento\AsynchronousOperations\Api\SaveMultipleOperationsInterface;
use Magento\AsynchronousOperations\Model\OperationManagement;
use Magento\Framework\Bulk\BulkManagementInterface;
use Magento\Framework\MessageQueue\ConnectionTypeResolver;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\SaaSOrderSync\Api\OrderSync\Bulk\OperationData;
use Magento\SaaSOrderSync\Api\Result;
use Psr\Log\LoggerInterface;

class BulkAdapter
{
    private LoggerInterface $logger;
    private SerializerInterface $serializer;
    private BulkManagementInterface $bulkManagement;
    private OperationInterfaceFactory $operationFactory;
    private OperationManagement $operationManagement;
    private SaveMultipleOperationsInterface $saveMultipleOperations;
    private BulkStatusInterface $bulkStatus;
    private ConnectionTypeResolver $connectionTypeResolver;

    public function __construct(
        LoggerInterface                 $logger,
        SerializerInterface             $serializer,
        BulkManagementInterface         $bulkManagement,
        OperationInterfaceFactory       $operationFactory,
        OperationManagement             $operationManagement,
        SaveMultipleOperationsInterface $saveMultipleOperations,
        BulkStatusInterface             $bulkStatus,
        ConnectionTypeResolver          $connectionTypeResolver,
    ) {
        $this->logger = $logger;
        $this->serializer = $serializer;
        $this->bulkManagement = $bulkManagement;
        $this->operationFactory = $operationFactory;
        $this->operationManagement = $operationManagement;
        $this->saveMultipleOperations = $saveMultipleOperations;
        $this->bulkStatus = $bulkStatus;
        $this->connectionTypeResolver = $connectionTypeResolver;
    }

    public function isAMQPAvailable(): bool
    {
        try {
            // amqp connection is defined in etc/queue_topology.xml
            $this->connectionTypeResolver->getConnectionType('amqp');
            return true;
        } catch (LogicException) {
            return false;
        }
    }

    public function scheduleOperation(array $bulk, $operationId, OperationData $operationData): Result
    {
        ['uuid' => $bulkUuid, 'description' => $bulkDescription] = $bulk;

        $operationData = [
            'data' => [
                'operation_key' => $operationId,
                'bulk_uuid' => $bulkUuid,
                'topic_name' => 'saas.order.sync.bulk',
                'serialized_data' => $this->serializer->serialize($operationData->getData()),
                'status' => OperationInterface::STATUS_TYPE_OPEN,
            ],
        ];
        $operation = $this->operationFactory->create($operationData);

        $scheduled = $this->bulkManagement->scheduleBulk($bulkUuid, [$operation], $bulkDescription);
        if (!$scheduled) {
            return Result::error('SCHEDULING_OPERATION_ERROR');
        }

        $this->saveMultipleOperations->execute([$operation]);
        return Result::data($operation);
    }

    public function failOperation(OperationInterface $operation, string $message, ?array $error = []): void
    {
        $this->operationManagement->changeOperationStatus(
            $operation->getBulkUuid(),
            $operation->getId(),
            OperationInterface::STATUS_TYPE_NOT_RETRIABLY_FAILED,
            message: __($message),
            data: $operation->getSerializedData(),
            resultData: $error ? $this->serializer->serialize($error) : null
        );
    }

    public function completeOperation(OperationInterface $operation, ?array $data = []): void
    {
        $this->operationManagement->changeOperationStatus(
            $operation->getBulkUuid(),
            $operation->getId(),
            OperationInterface::STATUS_TYPE_COMPLETE,
            resultData: $data ? $this->serializer->serialize($data) : null
        );
    }

    public function cancelOperation(OperationInterface $operation): void
    {
        $this->operationManagement->changeOperationStatus(
            $operation->getBulkUuid(),
            $operation->getId(),
            OperationInterface::STATUS_TYPE_REJECTED,
            message: __('Skip operation since order sync is canceled.'),
            data: $operation->getSerializedData()
        );
    }

    public function allOperationsCompleted(string $bulkUuid): bool
    {
        $operationList = $this->bulkStatus->getBulkShortStatus($bulkUuid)->getOperationsList();

        foreach ($operationList as $operation) {
            if ($operation->getStatus() == OperationInterface::STATUS_TYPE_OPEN ||
                $operation->getStatus() == OperationInterface::STATUS_TYPE_REJECTED) {
                return false;
            }
        }
        return true;
    }
}
