<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
namespace Magento\SaaSOrderSync\Test\Unit\Core\OrderSync;

use DateTime;
use Exception;
use Magento\AsynchronousOperations\Model\Operation;
use Magento\AsynchronousOperations\Model\OperationStatusPool;
use Magento\AsynchronousOperations\Model\OperationStatusValidator;
use Magento\Framework\Bulk\OperationInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\SaaSOrderSync\Api\OrderSync\Bulk\OperationProcessorPool;
use Magento\SaaSOrderSync\Api\Result;
use Magento\SaaSOrderSync\Core\Bulk\BulkAdapter;
use Magento\SaaSOrderSync\Core\CommerceDataExport\OrdersDataExporterAdapter;
use Magento\SaaSOrderSync\Core\OrderSync\Bulk\OperationCancelationPlugin;
use Magento\SaaSOrderSync\Core\OrderSync\Bulk\OperationCompletionPlugin;
use Magento\SaaSOrderSync\Core\OrderSync\Bulk\OperationConsumer;
use Magento\SaaSOrderSync\Core\OrderSync\Bulk\SyncOrderAggregatesOperationProcessor;
use Magento\SaaSOrderSync\Core\OrderSync\Config;
use Magento\SaaSOrderSync\Core\OrderSync\LocalOrderSync;
use Magento\SaaSOrderSync\Core\OrderSync\LocalOrderSyncRepository;
use Magento\SaaSOrderSync\Core\OrderSync\Lock;
use Magento\SaaSOrderSync\Core\OrderSync\SaaSClient;
use Magento\SaaSOrderSync\Test\Unit\Core\SaaS\SaaSClientMockResolver;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Log\NullLogger;
use function PHPUnit\Framework\anything;
use function PHPUnit\Framework\assertFalse;
use function PHPUnit\Framework\assertTrue;
use function PHPUnit\Framework\callback;
use function PHPUnit\Framework\never;
use function PHPUnit\Framework\once;
use function uniqid;

class OrdersBulkConsumerTest extends TestCase
{
    private OrdersDataExporterAdapter&MockObject $exporterAdapterMock;
    private BulkAdapter&MockObject $bulkAdapterMock;
    private SaaSClient&MockObject $orderSyncSaaSClient;
    private LocalOrderSyncRepository&MockObject $localOrderSyncRepository;

    private OrdersBulkConsumerWithPluginsRunner $orderBulkConsumerWithPlugins;

    public function setUp(): void
    {
        $logger = new NullLogger();
        $this->exporterAdapterMock = $this->createMock(OrdersDataExporterAdapter::class);
        $this->bulkAdapterMock = $this->createMock(BulkAdapter::class);
        $clientResolver = new SaaSClientMockResolver($this);
        $this->orderSyncSaaSClient = $clientResolver->createOrderSyncClient();
        $this->localOrderSyncRepository = $this->createMock(LocalOrderSyncRepository::class);

        $config = $this->createMock(Config::class);
        $config->method('getOperationSyncRequestOrderCount')->willReturn(100);
        $config->method('getOperationOrderCount')->willReturn(100);

        $ordersBulkConsumer = new OperationConsumer(
            $logger,
            new Json(),
            $this->bulkAdapterMock,
            new OperationProcessorPool([
                new SyncOrderAggregatesOperationProcessor($this->exporterAdapterMock, $clientResolver),
            ]),
            $config
        );

        $cancelationPlugin = new OperationCancelationPlugin(
            $logger,
            $this->localOrderSyncRepository,
            $this->bulkAdapterMock
        );

        $completionPlugin = new OperationCompletionPlugin(
            $logger,
            new Lock($logger, new NoLock()),
            $this->bulkAdapterMock,
            $clientResolver,
            $this->localOrderSyncRepository
        );

        $this->orderBulkConsumerWithPlugins = new OrdersBulkConsumerWithPluginsRunner(
            $ordersBulkConsumer,
            $cancelationPlugin,
            $completionPlugin
        );
    }

    public function test_shouldCompleteOperation_whenOrderSyncIsSuccessful()
    {
        $syncId = __METHOD__;
        $operationId = 1;
        $operation = $this->bulkOperation($syncId, $operationId, '{"orderIds": [1, 2, 3, 4, 5]}');
        $orderAggregates = [
            $this->commerceOrderAggregate(1),
            $this->commerceOrderAggregate(2),
            $this->commerceOrderAggregate(3),
            $this->commerceOrderAggregate(4),
            $this->commerceOrderAggregate(5),
        ];

        $this->exporterAdapterMock->expects(once())
            ->method('assignUuidsToOrderEntities')
            ->with([1, 2, 3, 4, 5])
            ->willReturn(0);

        $this->exporterAdapterMock->expects(once())
            ->method('exportCommerceOrderAggregates')
            ->with([1, 2, 3, 4, 5])
            ->willReturn($orderAggregates);

        $responseData = [
            'syncId' => $syncId,
            'processedCount' => 5,
            'errorCount' => 0,
        ];
        $this->orderSyncSaaSClient->expects(once())
            ->method('syncOrderBatch')
            ->with($syncId, $orderAggregates)
            ->willReturn(Result::data($responseData));

        $operationData = [
            'responses' => [
                0 => [
                    SyncOrderAggregatesOperationProcessor::class => [
                        'syncId' => $syncId,
                        'processedCount' => 5,
                        'errorCount' => 0,
                    ],
                ],
            ],
        ];
        $this->bulkAdapterMock->expects(once())
            ->method('completeOperation')
            ->with($operation, callback(function ($subject) use ($operationData) {
                return $subject['responses'] == $operationData['responses'] && $subject['processing_time_ms'] > 0;
            }));

        $success = $this->orderBulkConsumerWithPlugins->consume($operation);

        assertTrue($success);
    }

    public function test_shouldFailOperation_whenFailsOnOrdersDataExport()
    {
        $syncId = __METHOD__;
        $operationId = 2;
        $operation = $this->bulkOperation($syncId, $operationId, '{"orderIds": [1]}');

        $this->exporterAdapterMock->expects(once())
            ->method('exportCommerceOrderAggregates')
            ->willThrowException(new Exception('expected'));

        $this->orderSyncSaaSClient->expects(never())
            ->method(anything());

        $this->bulkAdapterMock->expects(once())
            ->method('failOperation')
            ->with($operation, anything(), anything());

        $success = $this->orderBulkConsumerWithPlugins->consume($operation);

        assertFalse($success);
    }

    public function test_shouldFailOperation_whenFailsOnSynchronizingOrders()
    {
        $syncId = __METHOD__;
        $operationId = 3;
        $operation = $this->bulkOperation($syncId, $operationId, '{"orderIds": [1, 2]}');
        $orderAggregates = [
            $this->commerceOrderAggregate(1),
            $this->commerceOrderAggregate(2),
        ];

        $this->exporterAdapterMock->expects(once())
            ->method('exportCommerceOrderAggregates')
            ->with([1, 2])
            ->willReturn($orderAggregates);

        $this->orderSyncSaaSClient->expects(once())
            ->method('syncOrderBatch')
            ->with($syncId, $orderAggregates)
            ->willThrowException(new Exception('expected'));

        $this->bulkAdapterMock->expects(once())
            ->method('failOperation')
            ->with($operation, anything(), anything());

        $success = $this->orderBulkConsumerWithPlugins->consume($operation);

        assertFalse($success);
    }

    public function test_shouldCancelOperation_whenSyncIsCancelled()
    {
        $syncId = __METHOD__;
        $operationId = 4;
        $operation = $this->bulkOperation($syncId, $operationId);

        $this->localOrderSyncRepository->expects(once())
            ->method('find')
            ->with($syncId)
            ->willReturn(new LocalOrderSync($syncId, new DateTime('2023-01-1')));

        $this->exporterAdapterMock->expects(never())
            ->method(anything());

        $this->orderSyncSaaSClient->expects(never())
            ->method(anything());

        $this->bulkAdapterMock->expects(once())
            ->method('cancelOperation')
            ->with($operation);

        $success = $this->orderBulkConsumerWithPlugins->consume($operation);

        assertFalse($success);
    }

    public function test_shouldCompleteSync_whenIsLastOperation()
    {
        $syncId = __METHOD__;
        $operationId = 5;
        $operation = $this->bulkOperation($syncId, $operationId, '{"orderIds": []}');

        $this->bulkAdapterMock->expects(once())
            ->method('allOperationsCompleted')
            ->with($syncId)
            ->willReturn(true);

        $this->orderSyncSaaSClient->expects(once())
            ->method('completeOrderSync')
            ->with($syncId)
            ->willReturn(Result::data());

        $success = $this->orderBulkConsumerWithPlugins->consume($operation);

        assertTrue($success);
    }

    private function commerceOrderAggregate($orderNumber): array
    {
        return ['commerceOrderNumber' => $orderNumber, 'orderId' => ['id' => uniqid()]];
    }

    private function bulkOperation(string $syncId, int $operationId, string $data = ''): Operation
    {
        $dummyValidator = new OperationStatusValidator(new OperationStatusPool());

        return new Operation($dummyValidator, [
            'operation_key' => $operationId,
            'bulk_uuid' => $syncId,
            'serialized_data' => $data,
            'status' => OperationInterface::STATUS_TYPE_OPEN,
        ]);
    }
}
