<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSOrderSync\Core\OrderSync;

use DateTime;
use DateTimeInterface;
use Magento\SaaSOrderSync\Api\Result;
use Magento\SaaSOrderSync\Core\SaaS\SaaSClient as Client;
use Psr\Http\Message\ResponseInterface;
use RuntimeException;

class SaaSClient
{
    private string $syncUrl;
    private Client $client;

    public function __construct(Client $client, string $syncUrl)
    {
        $this->syncUrl = $syncUrl;
        $this->client = $client;
    }

    public function createOrderSync(
        string   $syncId,
        DateTime $createdFrom,
        DateTime $createdTo,
        int      $initialCount,
        string   $clientCode,
        bool     $forceUpdate = false,
    ): array {
        $args = get_defined_vars();

        $body = [
            'syncId' => $syncId,
            'createdFrom' => $createdFrom->format(DateTimeInterface::ATOM),
            'createdTo' => $createdTo->format(DateTimeInterface::ATOM),
            'initialCount' => $initialCount,
            'clientCode' => $clientCode,
            'forceUpdate' => $forceUpdate,
        ];

        $res = $this->client->request('POST', $this->syncUrl, ['json' => $body]);

        if ($this->isSuccessful($res)) {
            return $this->body($res);
        }

        $this->throwException(__METHOD__, $args, $res);
    }

    public function syncOrderBatch(string $syncId, array $orderBatch): Result
    {
        $url = "$this->syncUrl/$syncId/orders";
        $body = [
            'orderAggregates' => $orderBatch,
        ];

        $res = $this->client->request('POST', $url, ['json' => $body]);
        return match ($res->getStatusCode()) {
            200 => Result::data($this->body($res)),
            404 => Result::error('NOT_FOUND'),
            409 => Result::error('NOT_IN_PROGRESS'),
            default => Result::error('INTERNAL', [
                'statusCode' => $res->getStatusCode(),
                'bodyContents' => $res->getBody()->getContents(),
            ]),
        };
    }

    public function getOneOrderSync(string $syncId): ?array
    {
        $args = get_defined_vars();

        $url = "$this->syncUrl/$syncId";
        $res = $this->client->request('GET', $url);

        return match (true) {
            $this->isSuccessful($res) => $this->body($res),
            $res->getStatusCode() == 404 => null,
            default => $this->throwException(__METHOD__, $args, $res)
        };
    }

    public function getAllOrderSyncs(): array
    {
        $args = get_defined_vars();

        $res = $this->client->request('GET', $this->syncUrl);

        if ($this->isSuccessful($res)) {
            return $this->body($res);
        }

        $this->throwException(__METHOD__, $args, $res);
    }

    public function completeOrderSync(string $syncId): Result
    {
        $url = "$this->syncUrl/$syncId/complete";
        $res = $this->client->request('PUT', $url);

        return match ($res->getStatusCode()) {
            200 => Result::data($this->body($res)),
            404 => Result::error('NOT_FOUND'),
            409 => Result::error('NOT_IN_PROGRESS'),
            default => Result::error('INTERNAL', [
                'statusCode' => $res->getStatusCode(),
                'bodyContents' => $res->getBody()->getContents(),
            ]),
        };
    }

    public function cancelOrderSync(string $syncId): Result
    {
        $url = "$this->syncUrl/$syncId/cancel";
        $res = $this->client->request('PUT', $url);

        return match ($res->getStatusCode()) {
            200 => Result::data($this->body($res)),
            404 => Result::error('NOT_FOUND'),
            409 => Result::error('NOT_IN_PROGRESS'),
            default => Result::error('INTERNAL', [
                'statusCode' => $res->getStatusCode(),
                'bodyContents' => $res->getBody()->getContents(),
            ]),
        };
    }

    public function syncRMABatch(string $syncId, array $rmaBatch): Result
    {
        $url = "$this->syncUrl/$syncId/rma";
        $body = [
            'rmaAggregates' => $rmaBatch,
        ];

        $res = $this->client->request('POST', $url, ['json' => $body]);
        return match ($res->getStatusCode()) {
            200 => Result::data($this->body($res)),
            404 => Result::error('NOT_FOUND'),
            409 => Result::error('NOT_IN_PROGRESS'),
            default => Result::error('INTERNAL', [
                'statusCode' => $res->getStatusCode(),
                'bodyContents' => $res->getBody()->getContents(),
            ]),
        };
    }

    private function isSuccessful(ResponseInterface $response): bool
    {
        return $response->getStatusCode() >= 200 && $response->getStatusCode() <= 299;
    }

    private function throwException(string $method, array $req, ResponseInterface $response)
    {
        $ctx = json_encode([
            'request' => $req,
            'response' => [
                'statusCode' => $response->getStatusCode(),
                'body' => $response->getBody()->getContents(),
            ],
        ]);
        throw new RuntimeException("Unexpected error on $method. ctx=$ctx");
    }

    private function body(ResponseInterface $res): array
    {
        return json_decode($res->getBody()->getContents(), true);
    }
}
