<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSOrderSync\Core\CommerceDataExport;

use DateTime;
use Generator;
use Magento\DataExporter\Export\Processor as ExportProcessor;
use Magento\DataExporter\Model\Indexer\FeedIndexMetadata;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Query\BatchIteratorFactory;
use Magento\SaaSOrderSync\Core\OrderSync\Config;
use Magento\SaaSOrderSync\Core\SaaS\SaaSConfig;
use Magento\SalesOrdersDataExporter\Model\OrderUuidManager;

class OrdersDataExporterAdapter
{
    private FeedIndexMetadata $metadata;
    private OrderUuidManager $uuidManager;
    private ResourceConnection $resourceConnection;
    private ExportProcessor $exportProcessor;
    private BatchIteratorFactory $batchIteratorFactory;
    private SaaSConfig $saaSConfig;

    private Config $config;

    public function __construct(
        FeedIndexMetadata    $metadata,
        ResourceConnection   $resourceConnection,
        ExportProcessor      $exportProcessor,
        BatchIteratorFactory $batchIteratorFactory,
        SaaSConfig           $saaSConfig,
        OrderUuidManager     $uuidManager,
        Config $config,
    ) {
        $this->metadata = $metadata;
        $this->resourceConnection = $resourceConnection;
        $this->exportProcessor = $exportProcessor;
        $this->batchIteratorFactory = $batchIteratorFactory;
        $this->saaSConfig = $saaSConfig;
        $this->uuidManager = $uuidManager;
        $this->config = $config;
    }

    public function count(DateTime $from, DateTime $to): int
    {
        $tableName = $this->resourceConnection->getTableName($this->metadata->getSourceTableName());
        $connection = $this->resourceConnection->getConnection();

        $select = $connection->select()
            ->from(
                ['order' => $tableName],
                ['count' => 'COUNT(entity_id)']
            )
            ->where('order.created_at >= ?', $from)
            ->where('order.created_at <= ?', $to);

        $row = $connection->fetchRow($select);
        return (int)$row['count'];
    }

    public function assignUuidsToOrderEntities(array $orderIds): int
    {
        return $this->uuidManager->assignByOrderIds($this->metadata->getBatchSize(), $orderIds);
    }

    /**
     * @link https://git.corp.adobe.com/magento/proto-schemas/blob/main/order/src/main/proto/com/adobe/commerce/sales/order/aggregate/v1/commerce_order_aggregate.proto CommerceOrderAggregate proto schema
     * @param array $orderIds
     * @return array of CommerceOrderAggregate
     */
    public function exportCommerceOrderAggregates(array $orderIds): array
    {
        $fieldName = $this->metadata->getFeedIdentity();
        $exportIds = array_map(fn ($orderId) => [$fieldName => $orderId], $orderIds);

        $orders = $this->exportProcessor->process($this->metadata->getFeedName(), $exportIds);
        return array_map($this->addMerchantInfo(), $orders);
    }

    public function findOrderIds(DateTime $from, DateTime $to): Generator
    {
        $connection = $this->resourceConnection->getConnection();
        $tableName = $this->resourceConnection->getTableName($this->metadata->getSourceTableName());
        $field = $this->metadata->getSourceTableField();
        $select = $connection->select()
            ->from($tableName, [$field])
            ->where('created_at >= ?', $from)
            ->where('created_at <= ?', $to);

        $batchSize = $this->config->getNumberOfAggregatesInOperation();
        $iterator = $this->batchIteratorFactory->create([
            'select' => $select,
            'batchSize' => $batchSize,
            'correlationName' => $tableName,
            'rangeField' => $field,
            'rangeFieldAlias' => $field,
        ]);

        foreach ($iterator as $batchSelect) {
            yield $connection->fetchCol($batchSelect);
        }
    }

    private function addMerchantInfo(): callable
    {
        $merchantInfo = [
            'mageId' => $this->saaSConfig->getMageId(),
            'environmentId' => $this->saaSConfig->getEnvironmentId(),
        ];

        return function (array $order) use ($merchantInfo): array {
            $order['merchantInfo'] = $merchantInfo;
            return $order;
        };
    }
}
