<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSOrderSync\Console\Command;

use Magento\AsynchronousOperations\Api\BulkStatusInterface;
use Magento\AsynchronousOperations\Api\Data\OperationInterface;
use Magento\Framework\Bulk\BulkSummaryInterface;
use Magento\Framework\Console\Cli;
use Magento\Framework\Webapi\Exception as WebapiException;
use Magento\SaaSOrderSync\Api\OrderSync\OrderSyncManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class GetOrderSyncStatus extends Command
{
    private const COMMAND_NAME = 'saas:order-sync:status';

    private OrderSyncManagerInterface $orderSyncManager;
    private BulkStatusInterface $bulkStatus;

    public function __construct(OrderSyncManagerInterface $orderSyncManager, BulkStatusInterface $bulkStatus)
    {
        $this->orderSyncManager = $orderSyncManager;
        $this->bulkStatus = $bulkStatus;
        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->setName(self::COMMAND_NAME)
            ->setDescription('Get the status of a SaaS order synchronization.')
            ->addArgument(
                'syncId',
                InputArgument::REQUIRED,
                'Sync id'
            )
            ->addOption('show-bulk-detail', 'b', InputOption::VALUE_NONE, "Show detail of the bulk.");
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $syncId = $input->getArgument('syncId');
        $showBulkDetail = $input->getOption('show-bulk-detail');

        try {
            $response = $this->orderSyncManager->getOneOrderSync($syncId);
            $sync = $response->getOrderSync();

            $data = [
                "syncId" => $sync->getSyncId(),
                "createdFrom" => $sync->getCreatedFrom(),
                "createdTo" => $sync->getCreatedTo(),
                "clientCode" => $sync->getClientCode(),
                "status" => $sync->getStatus(),
                "initialCount" => $sync->getInitialCount(),
                "processedCount" => $sync->getProcessedCount(),
                "errorCount" => $sync->getErrorCount(),
            ];

            if ($showBulkDetail) {
                $bulkId = $sync->getSyncId();
                $bulkDetail = $this->bulkStatus->getBulkDetailedStatus($bulkId);
                $data['bulk'] = [
                    'bulkId' => $bulkId,
                    'bulkStatus' => $this->formatBulkStatus($this->bulkStatus->getBulkStatus($bulkId)),
                    'operationCount' => $bulkDetail->getOperationCount(),
                    'operations' => array_map(
                        fn (OperationInterface $operation): array => [
                            'operationId' => $operation->getId(),
                            'status' => $this->formatOperationStatus((int)$operation->getStatus()),
                            'errorCode' => $operation->getErrorCode(),
                            'resultMessage' => $operation->getResultMessage(),
                            'resultSerializedData' => $operation->getResultSerializedData() ? json_decode($operation->getResultSerializedData()) : null
                        ],
                        array_values($bulkDetail->getOperationsList())
                    )
                ];
            }

            $pretty = json_encode($data, JSON_PRETTY_PRINT);
            $output->writeln("<info>$pretty</info>");

            return Cli::RETURN_SUCCESS;
        } catch (WebapiException $e) {
            $output->writeln("<error>{$e->getHttpCode()}</error>");
            return Cli::RETURN_FAILURE;
        }
    }

    private function formatBulkStatus(int $bulkStatusId): string
    {
        $status = match ($bulkStatusId) {
            BulkSummaryInterface::NOT_STARTED => 'NOT_STARTED',
            BulkSummaryInterface::IN_PROGRESS => 'IN_PROGRESS',
            BulkSummaryInterface::FINISHED_SUCCESSFULLY => 'FINISHED_SUCCESSFULLY',
            BulkSummaryInterface::FINISHED_WITH_FAILURE => 'FINISHED_WITH_FAILURE',
            default => 'UNKNOWN'
        };

        return "$status ($bulkStatusId)";
    }

    private function formatOperationStatus(int $operationStatusId): string
    {

        $status = match ($operationStatusId) {
            OperationInterface::STATUS_TYPE_COMPLETE => 'COMPLETE',
            OperationInterface::STATUS_TYPE_RETRIABLY_FAILED => 'RETRIABLY_FAILED',
            OperationInterface::STATUS_TYPE_NOT_RETRIABLY_FAILED => 'NOT_RETRIABLY_FAILED',
            OperationInterface::STATUS_TYPE_OPEN => 'OPEN',
            OperationInterface::STATUS_TYPE_REJECTED => 'REJECTED',
            default => 'UNKNOWN'
        };

        return "$status ($operationStatusId)";
    }
}
