<?php

declare(strict_types=1);

namespace Magento\SaaSOrderSync\Api;

use Exception;
use Magento\Framework\DataObject;
use Throwable;

/**
 * Return type to model error and data results. Using array destructuring it is easy to build a construct
 * which is flexible enough to handle success cases but also multiple error cases.
 *
 * Example of usage:
 * ```
 * [$err, $data] = foo();
 * if ($err) {
 *   match($err['code']) {
 *     'foo' => 'bar',
 *     'default' => 'baz'
 *   }
 * } else {
 *   // handle data
 * }
 * ```
 */
class Result extends DataObject
{
    public static function error(string $code, array $context = []): Result
    {
        $err = ['code' => $code];
        if ($context) {
            $err['context'] = $context;
        }
        return new Result([$err, []]);
    }

    public static function data(mixed $data = null): Result
    {
        return new Result([[], $data]);
    }

    public static function wrap(Throwable|callable $arg): Result
    {
        if ($arg instanceof Exception) {
            return Result::error(
                'EXCEPTION',
                ['exception' => $arg]
            );
        }

        if ($arg instanceof Throwable) {
            return Result::error(
                'THROWABLE',
                ['throwable' => $arg]
            );
        }

        try {
            $ret = $arg();
            if ($ret instanceof Result) {
                return $ret;
            }
            return Result::data($ret);
        } catch (Throwable $e) {
            return Result::wrap($e);
        }
    }

    public function isData(): bool
    {
        return !$this->isError();
    }

    public function isError(): bool
    {
        return !empty($this[0]);
    }

    public function getErrorCode(): ?string
    {
        return $this[0]['code'];
    }

    public function __toString(): string
    {
        [$err, $data] = $this->_data;
        if ($err) {
            return json_encode(['error' => $err]);
        }
        return json_encode(['data' => $data]);
    }
}
