<?php

namespace Magento\SaaSOrderSync\Core\OrderSync\Bulk;

use Magento\AsynchronousOperations\Api\Data\OperationInterface;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\SaaSOrderSync\Api\OrderSync\Bulk\OperationData;
use Magento\SaaSOrderSync\Api\OrderSync\Bulk\OperationProcessorPool;
use Magento\SaaSOrderSync\Api\Result;
use Magento\SaaSOrderSync\Core\Bulk\BulkAdapter;
use Psr\Log\LoggerInterface;

class OperationConsumer
{
    private LoggerInterface $logger;
    private SerializerInterface $serializer;
    private BulkAdapter $bulkAdapter;
    private OperationProcessorPool $operationProcessorPool;

    public function __construct(
        LoggerInterface        $logger,
        SerializerInterface    $serializer,
        BulkAdapter            $bulkAdapter,
        OperationProcessorPool $operationProcessorPool
    ) {
        $this->logger = $logger;
        $this->serializer = $serializer;
        $this->bulkAdapter = $bulkAdapter;
        $this->operationProcessorPool = $operationProcessorPool;
    }

    public function consume(OperationInterface $operation): bool
    {
        $syncId = $operation->getBulkUuid();
        $operationId = $operation->getId();
        $ctx = ['syncId' => $syncId, 'operationId' => $operationId];
        $this->logger->debug('Order sync operation started.', $ctx);

        [$err, $data] = $this->process($operation);
        if ($err) {
            [$level, $message] = match ($err['code']) {
                'NOT_IN_PROGRESS' => ['WARNING', 'Operation has failed because order sync is not in progress.'],
                default => ['CRITICAL', 'Unexpected error during order sync operation.']
            };

            $this->bulkAdapter->failOperation($operation, "$level - $message", $err);

            $ctx['error'] = $err;
            $this->logger->log($level, $message, $ctx);
            return false;
        }

        $this->bulkAdapter->completeOperation($operation, $data);

        $this->logger->debug('Order sync operation finished.', $ctx);
        return true;
    }

    private function process(OperationInterface $operation): Result
    {
        return Result::wrap(function () use ($operation) {
            $ctx = ['syncId' => $operation->getBulkUuid(), 'operationId' => $operation->getId()];
            $operationData = new OperationData($this->serializer->unserialize($operation->getSerializedData()));
            $results = [];

            foreach ($this->operationProcessorPool->get() as $operationProcessor) {
                $class = $operationProcessor::class;

                $this->logger->debug("Start execution of operation processor $class.", $ctx);
                $result = $operationProcessor->process($operation->getBulkUuid(), $operationData);
                $this->logger->debug("End execution of operation processor $class.", array_merge($ctx, ['result' => $result]));

                [$err, $data] = $result;
                $results[$class] = $err ?: $data;
                if ($err) {
                    return Result::error($err['code'], $results);
                }
            }

            return $results;
        });
    }
}
