<?php

namespace Magento\SaaSOrderSync\Core\OrderSync;

use Magento\AsynchronousOperations\Api\Data\OperationInterface;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\SaaSOrderSync\Core\Bulk\BulkAdapter;
use Magento\SaaSOrderSync\Core\CommerceDataExport\OrdersDataExporterAdapter;
use Magento\SaaSOrderSync\Core\Common\Result;
use Magento\SaaSOrderSync\Core\SaaS\SaaSClientResolverInterface;
use Psr\Log\LoggerInterface;

class OrdersBulkConsumer
{
    private LoggerInterface $logger;
    private SerializerInterface $serializer;
    private BulkAdapter $bulkAdapter;
    private OrdersDataExporterAdapter $ordersExporterAdapter;
    private SaaSClientResolverInterface $clientResolver;

    public function __construct(
        LoggerInterface             $logger,
        SerializerInterface         $serializer,
        BulkAdapter                 $bulkAdapter,
        OrdersDataExporterAdapter   $ordersExporter,
        SaaSClientResolverInterface $clientResolver
    ) {
        $this->logger = $logger;
        $this->serializer = $serializer;
        $this->bulkAdapter = $bulkAdapter;
        $this->ordersExporterAdapter = $ordersExporter;
        $this->clientResolver = $clientResolver;
    }

    public function consume(OperationInterface $operation): bool
    {
        $syncId = $operation->getBulkUuid();
        $operationId = $operation->getId();
        $ctx = ['syncId' => $syncId, 'operationId' => $operationId];
        $this->logger->debug('Order sync operation started.', $ctx);

        [$err, $data] = $this->syncOrders($operation);

        if ($err) {
            [$level, $message] = match ($err['code']) {
                'NOT_IN_PROGRESS' => ['WARNING', 'Operation has failed because order sync is not in progress.'],
                default => ['CRITICAL', 'Unexpected error during order sync operation.']
            };

            $this->bulkAdapter->failOperation($operation, "$level - $message", $err);

            $ctx['error'] = $err;
            $this->logger->log($level, $message, $ctx);
            return false;
        }

        $this->bulkAdapter->completeOperation($operation, $data);

        $ctx['processedCount'] = $data['processedCount'] ?? 0;
        $ctx['errorCount'] = $data['errorCount'] ?? 0;
        $this->logger->debug('Order sync operation finished.', $ctx);
        return true;
    }

    private function syncOrders(OperationInterface $operation): Result
    {
        return Result::wrap(function () use ($operation) {
            ['orderIds' => $orderIds] = $this->serializer->unserialize($operation->getSerializedData());

            $this->ordersExporterAdapter->assignUuidsToOrderEntities($orderIds);
            $orders = $this->ordersExporterAdapter->exportCommerceOrderAggregates($orderIds);
            $syncClient = $this->clientResolver->createOrderSyncClient();

            $syncId = $operation->getBulkUuid();
            return $syncClient->syncOrderBatch($syncId, $orders);
        });
    }
}
