<?php

namespace Magento\SaaSOrderSync\Core\OrderSync;

use Magento\AsynchronousOperations\Api\Data\OperationInterface;
use Magento\SaaSOrderSync\Core\Bulk\BulkAdapter;
use Magento\SaaSOrderSync\Core\SaaS\SaaSClientResolverInterface;
use Psr\Log\LoggerInterface;
use Throwable;

class OrdersBulkCompletionPlugin
{
    private LoggerInterface $logger;
    private OrderSyncLock $orderSyncLock;
    private BulkAdapter $bulkAdapter;
    private SaaSClientResolverInterface $clientResolver;
    private LocalOrderSyncRepository $localOrderSyncRepository;

    public function __construct(
        LoggerInterface             $logger,
        OrderSyncLock               $lock,
        BulkAdapter                 $bulkStatus,
        SaaSClientResolverInterface $clientResolver,
        LocalOrderSyncRepository    $localOrderSyncRepository,
    ) {
        $this->logger = $logger;
        $this->orderSyncLock = $lock;
        $this->bulkAdapter = $bulkStatus;
        $this->clientResolver = $clientResolver;
        $this->localOrderSyncRepository = $localOrderSyncRepository;
    }

    /**
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function afterConsume($_, bool $result, OperationInterface $operation): bool
    {
        $syncId = $operation->getBulkUuid();
        $operationId = $operation->getId();

        $ctx = ['syncId' => $syncId, 'operationId' => $operationId];
        try {
            $this->orderSyncLock->locking($syncId, fn () => $this->complete($syncId, $operationId));
        } catch (Throwable $e) {
            $ctx['exception'] = $e;
            $this->logger->critical('Unexpected error in order sync complete plugin.', $ctx);
        }
        return $result;
    }

    public function complete(string $syncId, string $operationId): void
    {
        $ctx = ["syncId" => $syncId, "operationId" => $operationId];

        $localOrderSync = $this->localOrderSyncRepository->find($syncId);
        if ($localOrderSync && $localOrderSync->isCanceled()) {
            $this->logger->warning('Skip completion since order sync is canceled.', $ctx);
            return;
        }

        if (!$this->bulkAdapter->allOperationsCompleted($syncId)) {
            $this->logger->debug('Order sync is not completed.', $ctx);
            return;
        }

        $syncClient = $this->clientResolver->createOrderSyncClient();
        [$err, $data] = $syncClient->completeOrderSync($syncId);
        if ($err) {
            $errorCode = $err['code'];
            $level = match ($errorCode) {
                'NOT_FOUND', 'NOT_IN_PROGRESS' => 'WARNING',
                default => 'CRITICAL'
            };
            $ctx['error'] = $err;
            $this->logger->log($level, 'Unexpected error when completing order sync', $ctx);
            return;
        }

        $this->logger->info('Completed order sync.', $data);
    }
}
