<?php

declare(strict_types=1);

namespace Magento\SaaSOrderSync\Core\OrderSync;

use function __;
use DateTime;
use DateTimeInterface;
use Exception;
use Magento\Framework\DataObject\IdentityGeneratorInterface;
use Magento\Framework\Webapi\Exception as WebapiException;
use Magento\SaaSOrderSync\Api\OrderSync\CancelOrderSyncResponseInterface;
use Magento\SaaSOrderSync\Api\OrderSync\CreateOrderSyncResponseInterface;
use Magento\SaaSOrderSync\Api\OrderSync\GetAllOrderSyncsResponseInterface;
use Magento\SaaSOrderSync\Api\OrderSync\GetOneOrderSyncResponseInterface;
use Magento\SaaSOrderSync\Api\OrderSync\OrderSyncManagerInterface;
use Magento\SaaSOrderSync\Core\CommerceDataExport\OrdersDataExporterAdapter;
use Magento\SaaSOrderSync\Core\Common\Time\ClockInterface;
use Magento\SaaSOrderSync\Core\SaaS\SaaSClientResolverInterface;
use Psr\Log\LoggerInterface;
use Throwable;

class OrderSyncManager implements OrderSyncManagerInterface
{
    private IdentityGeneratorInterface $identityGenerator;
    private OrdersDataExporterAdapter $ordersExporter;
    private OrdersBulkManager $ordersBulkManager;
    private SaaSClientResolverInterface $clientResolver;
    private LoggerInterface $logger;
    private OrderSyncLock $orderSyncLock;
    private ClockInterface $clock;

    public function __construct(
        IdentityGeneratorInterface  $identityGenerator,
        OrdersDataExporterAdapter   $ordersExporter,
        OrdersBulkManager           $ordersBulkManager,
        SaaSClientResolverInterface $clientResolver,
        LoggerInterface             $logger,
        OrderSyncLock               $orderSyncLock,
        ClockInterface              $clock,
    ) {
        $this->identityGenerator = $identityGenerator;
        $this->ordersExporter = $ordersExporter;
        $this->ordersBulkManager = $ordersBulkManager;
        $this->clientResolver = $clientResolver;
        $this->logger = $logger;
        $this->orderSyncLock = $orderSyncLock;
        $this->clock = $clock;
    }

    public function createOrderSync(string $created_from, string $client_code, bool $optimize_time_range = true): CreateOrderSyncResponseInterface
    {
        $syncId = $this->identityGenerator->generateId();
        $ctx = ['syncId' => $syncId, 'createdFrom' => $created_from, 'clientCode' => $client_code];

        $createdFrom = $this->validateCreatedFrom($created_from);
        $clientCode = $this->parseString($client_code)
            ?: $this->throwBadRequest('client_code can\'t be empty.');

        try {
            $now = $this->clock->now();
            $syncTimeFrame = $this->resolveSyncTimeRange($createdFrom, $now, $optimize_time_range);
            if (empty($syncTimeFrame)) {
                return $this->createCompletedSync($syncId, $createdFrom, $now, $clientCode);
            }

            [$createdFrom, $createdTo] = $syncTimeFrame;

            $initialCount = $this->ordersExporter->count($createdFrom, $createdTo);
            if ($initialCount == 0) {
                return $this->createCompletedSync($syncId, $createdFrom, $createdTo, $clientCode);
            }

            $orderSyncClient = $this->clientResolver->createOrderSyncClient();
            $orderSync = $orderSyncClient->createOrderSync($syncId, $createdFrom, $createdTo, $initialCount, $clientCode);

            $this->ordersBulkManager->schedule($syncId, $createdTo, $createdFrom);

            $this->logger->info('Created SaaS order sync.', $ctx);

            return new CreateOrderSyncResponse($this->map($orderSync));
        } catch (WebapiException $e) {
            throw $e;
        } catch (Throwable $e) {
            $this->logAndRethrow($e, "Unexpected exception on " . (__METHOD__) . ".", $ctx);
        }
    }

    public function getAllOrderSyncs(): GetAllOrderSyncsResponseInterface
    {
        try {
            $orderSyncClient = $this->clientResolver->createOrderSyncClient();
            ['orderSyncs' => $orderSyncs] = $orderSyncClient->getAllOrderSyncs();

            // @formatter:off
            return new GetAllOrderSyncsResponse(array_map(
                fn (array $orderSync): OrderSync => $this->map($orderSync),
                $orderSyncs
            ));
            // @formatter:on
        } catch (Throwable $e) {
            $this->logAndRethrow($e, "Unexpected exception on " . (__METHOD__) . ".");
        }
    }

    public function getOneOrderSync($sync_id): GetOneOrderSyncResponseInterface
    {
        try {
            $orderSyncClient = $this->clientResolver->createOrderSyncClient();
            $orderSync = $orderSyncClient->getOneOrderSync($sync_id);

            if ($orderSync == null) {
                throw new WebapiException(__("No order sync found for id '$sync_id'."));
            }

            return new GetOneOrderSyncResponse($this->map($orderSync));
        } catch (WebapiException $e) {
            throw $e;
        } catch (Throwable $e) {
            $this->logAndRethrow($e, "Unexpected exception on " . (__METHOD__) . ".", ['syncId' => $sync_id]);
        }
    }

    public function cancelOrderSync(string $sync_id): CancelOrderSyncResponseInterface
    {
        try {
            return $this->orderSyncLock->locking($sync_id, function ($syncId) {
                $orderSyncClient = $this->clientResolver->createOrderSyncClient();
                [$err, $data] = $orderSyncClient->cancelOrderSync($syncId);

                if ($err) {
                    match ($err['code']) {
                        'NOT_FOUND' => throw new WebapiException(__("No order sync found for id '$syncId'.")),
                        'NOT_IN_PROGRESS' => throw new WebapiException(__("Order sync '$syncId' is not in progress."), httpCode: 409),
                        default => $this->logAndThrow(
                            'Unexpected error while canceling order sync in SaaS.',
                            ['syncId' => $syncId, 'error' => $err]
                        )
                    };
                }

                $this->ordersBulkManager->cancel($syncId);

                $this->logger->info('SaaS order sync has been canceled.', ['syncId' => $syncId]);

                return new CancelOrderSyncResponse($this->map($data));
            });
        } catch (WebapiException $e) {
            throw $e;
        } catch (Throwable $e) {
            $this->logAndRethrow($e, "Unexpected exception on " . (__METHOD__) . ".", ['syncId' => $sync_id]);
        }
    }

    private function map(array $orderSync): OrderSync
    {
        return new OrderSync(
            $orderSync['syncId'],
            $orderSync['createdFrom'],
            $orderSync['createdTo'],
            $orderSync['clientCode'],
            $orderSync['initialCount'],
            $orderSync['processedCount'],
            $orderSync['errorCount'],
            $orderSync['status']
        );
    }

    private function validateCreatedFrom(string $createdFrom): DateTime
    {
        $parsedCreatedFrom = $this->parseDateTime($createdFrom)
            ?: $this->throwBadRequest('created_from can\'t be parsed in ISO8601 format.');

        if ($this->clock->now() < $parsedCreatedFrom) {
            $this->throwBadRequest('created_from should be in the past.');
        }

        return $parsedCreatedFrom;
    }

    public function parseDateTime(string $dateTime): DateTime|bool
    {
        try {
            return new DateTime($dateTime);
        } catch (Exception) {
            return false;
        }
    }

    private function parseString(string $string): string|bool
    {
        return strlen(trim($string)) === 0 ? false : $string;
    }

    private function resolveSyncTimeRange(DateTime $createdFrom, DateTime $now, bool $optimize_time_range): array
    {
        if (!$optimize_time_range) {
            return [$createdFrom, $now];
        }

        $merchantClient = $this->clientResolver->createMerchantClient();
        ['orders' => $ordersDataSummary] = $merchantClient->getDataSummary();
        if (!array_key_exists('createdFrom', $ordersDataSummary)) {
            return [$createdFrom, $now];
        }

        $createdTo = $this->parseDateTime($ordersDataSummary['createdFrom']);
        if (!$createdTo) {
            $this->logAndThrow(
                'orders created_from of merchant data summary can\'t be parsed in ISO8601 format.',
                ['ordersDataSummary' => $ordersDataSummary]
            );
        }

        if ($createdFrom >= $createdTo) {
            $this->logger->debug('Order sync time range'
                . " from {$createdFrom->format(DateTimeInterface::ATOM)}"
                . " to {$createdTo->format(DateTimeInterface::ATOM)}"
                . ' is already synced.');
            return [];
        }

        return [$createdFrom, $createdTo];
    }

    private function createCompletedSync(string $syncId, DateTime $createdFrom, DateTime $createdTo, string $clientCode): CreateOrderSyncResponseInterface
    {
        $orderSyncClient = $this->clientResolver->createOrderSyncClient();
        $orderSyncClient->createOrderSync($syncId, $createdFrom, $createdTo, 0, $clientCode);
        [$err, $data] = $orderSyncClient->completeOrderSync($syncId);
        if ($err) {
            $this->logAndThrow('Failed to complete order sync.', ['syncId' => $syncId]);
        }

        return new CreateOrderSyncResponse($this->map($data));
    }

    private function logAndThrow(string $msg, array $ctx = [])
    {
        $e = new WebapiException(
            __($msg),
            httpCode: WebapiException::HTTP_INTERNAL_ERROR
        );
        $this->logAndRethrow($e, $msg, $ctx);
    }

    private function logAndRethrow(Throwable $e, string $msg, array $ctx = [])
    {
        $ctx['exception'] = $e;
        $this->logger->critical($msg, $ctx);
        throw $e;
    }

    public function throwBadRequest($msg)
    {
        throw new WebapiException(
            __($msg),
            httpCode: WebapiException::HTTP_BAD_REQUEST,
        );
    }
}
