<?php

namespace Magento\SaaSOrderSync\Core\Bulk;

use Magento\AsynchronousOperations\Api\BulkStatusInterface;
use Magento\AsynchronousOperations\Api\Data\OperationInterface;
use Magento\AsynchronousOperations\Api\Data\OperationInterfaceFactory;
use Magento\AsynchronousOperations\Api\SaveMultipleOperationsInterface;
use Magento\AsynchronousOperations\Model\OperationManagement;
use Magento\Framework\Bulk\BulkManagementInterface;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\SaaSOrderSync\Core\Common\Result;
use Psr\Log\LoggerInterface;

class BulkAdapter
{
    private LoggerInterface $logger;
    private SerializerInterface $serializer;
    private BulkManagementInterface $bulkManagement;
    private OperationInterfaceFactory $operationFactory;
    private OperationManagement $operationManagement;
    private SaveMultipleOperationsInterface $saveMultipleOperations;
    private BulkStatusInterface $bulkStatus;

    public function __construct(
        LoggerInterface                 $logger,
        SerializerInterface             $serializer,
        BulkManagementInterface         $bulkManagement,
        OperationInterfaceFactory       $operationFactory,
        OperationManagement             $operationManagement,
        SaveMultipleOperationsInterface $saveMultipleOperations,
        BulkStatusInterface             $bulkStatus,
    ) {
        $this->logger = $logger;
        $this->serializer = $serializer;
        $this->bulkManagement = $bulkManagement;
        $this->operationFactory = $operationFactory;
        $this->operationManagement = $operationManagement;
        $this->saveMultipleOperations = $saveMultipleOperations;
        $this->bulkStatus = $bulkStatus;
    }

    public function scheduleOperation(array $bulk, $operationId, array $operationData): Result
    {
        ['uuid' => $bulkUuid, 'description' => $bulkDescription] = $bulk;

        $operationData = [
            'data' => [
                'operation_key' => $operationId,
                'bulk_uuid' => $bulkUuid,
                'topic_name' => 'saas.order.sync.bulk',
                'serialized_data' => $this->serializer->serialize($operationData),
                'status' => OperationInterface::STATUS_TYPE_OPEN,
            ],
        ];
        $operation = $this->operationFactory->create($operationData);

        $scheduled = $this->bulkManagement->scheduleBulk($bulkUuid, [$operation], $bulkDescription);
        if (!$scheduled) {
            return Result::error('SCHEDULING_OPERATION_ERROR');
        }

        $this->saveMultipleOperations->execute([$operation]);
        return Result::data($operation);
    }

    public function failOperation(OperationInterface $operation, string $message, ?array $error = []): void
    {
        $this->operationManagement->changeOperationStatus(
            $operation->getBulkUuid(),
            $operation->getId(),
            OperationInterface::STATUS_TYPE_NOT_RETRIABLY_FAILED,
            message: __($message),
            data: $operation->getSerializedData(),
            resultData: $error ? $this->serializer->serialize($error) : null
        );
    }

    public function completeOperation(OperationInterface $operation, ?array $data = []): void
    {
        $this->operationManagement->changeOperationStatus(
            $operation->getBulkUuid(),
            $operation->getId(),
            OperationInterface::STATUS_TYPE_COMPLETE,
            resultData: $data ? $this->serializer->serialize($data) : null
        );
    }

    public function cancelOperation(OperationInterface $operation): void
    {
        $this->operationManagement->changeOperationStatus(
            $operation->getBulkUuid(),
            $operation->getId(),
            OperationInterface::STATUS_TYPE_REJECTED,
            message: __('Skip operation since order sync is canceled.'),
            data: $operation->getSerializedData()
        );
    }

    public function allOperationsCompleted(string $bulkUuid): bool
    {
        $operationList = $this->bulkStatus->getBulkShortStatus($bulkUuid)->getOperationsList();

        foreach ($operationList as $operation) {
            if ($operation->getStatus() == OperationInterface::STATUS_TYPE_OPEN ||
                $operation->getStatus() == OperationInterface::STATUS_TYPE_REJECTED) {
                return false;
            }
        }
        return true;
    }
}
