<?php
declare(strict_types=1);

namespace Magento\SaaSOrderSync\Console\Command;

use Magento\Framework\Console\Cli;
use Magento\Framework\Webapi\Exception as WebapiException;
use Magento\SaaSOrderSync\Api\OrderSync\OrderSyncManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class StartOrderSync extends Command
{
    private const COMMAND_NAME = 'saas:order-sync:start';

    private OrderSyncManagerInterface $orderSyncManager;

    public function __construct(OrderSyncManagerInterface $orderSyncManager)
    {
        $this->orderSyncManager = $orderSyncManager;
        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->setName(self::COMMAND_NAME)
            ->setDescription('Synchronize orders with the SaaS from a certain point in time.')
            ->addArgument(
                'from',
                InputArgument::REQUIRED,
                'From date time in ISO-8601 format.'
            )
            ->addOption(
                'optimize-time-range',
                'o',
                InputOption::VALUE_OPTIONAL,
                'Optimize requested time range by inspecting the merchant data summary (true|false).',
                'true',
            );
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $createdFrom = $input->getArgument('from');

        $optimizeTimeRange = $input->getOption('optimize-time-range') === 'true';
        $clientCode = 'cli';

        try {
            $response = $this->orderSyncManager->createOrderSync($createdFrom, $clientCode, $optimizeTimeRange);
            $sync = $response->getOrderSync();

            $data = [
                "syncId" => $sync->getSyncId(),
                "createdFrom" => $sync->getCreatedFrom(),
                "createdTo" => $sync->getCreatedTo(),
                "clientCode" => $sync->getClientCode(),
                "status" => $sync->getStatus(),
                "initialCount" => $sync->getInitialCount(),
                "processedCount" => $sync->getProcessedCount(),
                "errorCount" => $sync->getErrorCount(),
            ];

            $pretty = json_encode($data, JSON_PRETTY_PRINT);
            $output->writeln("<info>$pretty</info>");

            return Cli::RETURN_SUCCESS;
        } catch (WebapiException $e) {
            $msg = $e->getHttpCode() < 500
                ? $e->getMessage()
                : "Unexpected error while executing the command. Check the logs.";

            $output->writeln("<error>$msg</error>");
            return Cli::RETURN_FAILURE;
        }
    }
}
