<?php
declare(strict_types=1);

namespace Magento\SaaSOrderSync\Console\Command;

use Magento\Framework\Console\Cli;
use Magento\Framework\Webapi\Exception as WebapiException;

use Magento\SaaSOrderSync\Api\OrderSync\OrderSyncManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

class CancelOrderSync extends Command
{
    private const COMMAND_NAME = 'saas:order-sync:cancel';

    private OrderSyncManagerInterface $orderSyncManager;

    public function __construct(OrderSyncManagerInterface $orderSyncManager)
    {
        $this->orderSyncManager = $orderSyncManager;
        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->setName(self::COMMAND_NAME)
            ->setDescription('Cancel a SaaS order synchronization.')
            ->addArgument(
                'syncId',
                InputArgument::REQUIRED,
                'Sync id'
            );
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $syncId = $input->getArgument('syncId');

        try {
            $response = $this->orderSyncManager->cancelOrderSync($syncId);
            $sync = $response->getOrderSync();

            $data = [
                "syncId" => $sync->getSyncId(),
                "createdFrom" => $sync->getCreatedFrom(),
                "createdTo" => $sync->getCreatedTo(),
                "clientCode" => $sync->getClientCode(),
                "status" => $sync->getStatus(),
                "initialCount" => $sync->getInitialCount(),
                "processedCount" => $sync->getProcessedCount(),
                "errorCount" => $sync->getErrorCount(),
            ];

            $pretty = json_encode($data, JSON_PRETTY_PRINT);
            $output->writeln("<info>$pretty</info>");

            return Cli::RETURN_SUCCESS;
        } catch (WebapiException $e) {
            $msg = $e->getHttpCode() < 500
                ? $e->getMessage()
                : "Unexpected error while executing the command. Check the logs.";

            $output->writeln("<error>$msg</error>");
            return Cli::RETURN_FAILURE;
        }
    }
}
