<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSCustomerSync\Console\Command;

use Magento\AsynchronousOperations\Api\BulkStatusInterface;
use Magento\AsynchronousOperations\Api\Data\OperationInterface;
use Magento\Framework\Bulk\BulkSummaryInterface;
use Magento\Framework\Console\Cli;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Webapi\Exception as WebapiException;
use Magento\SaaSCustomerSync\Api\CustomerSyncManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class GetCustomerSync extends Command
{
    private const COMMAND_NAME = 'saas:customer-sync:get';

    public function __construct(
        private readonly CustomerSyncManagerInterface $customerSyncManager,
        private readonly BulkStatusInterface          $bulkStatus,
    ) {
        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->setName(self::COMMAND_NAME)
            ->setDescription('Get the customer synchronization.')
            ->addArgument('sync_id', InputArgument::REQUIRED, 'The synchronization id.')
            ->addOption('show-bulk-detail', 'b', InputOption::VALUE_NONE, "Show detail of the bulk related to the synchronization.");
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $syncId = $input->getArgument('sync_id');
        $showBulkDetail = $input->getOption('show-bulk-detail');
        try {
            $response = $this->customerSyncManager->getCustomerSync($syncId);

            $data = [
                "syncId" => $response->getSyncId(),
                "status" => $this->formatBulkStatus($response->getStatus()),
            ];

            if ($showBulkDetail) {
                $bulkId = $response->getSyncId();
                $bulkDetail = $this->bulkStatus->getBulkDetailedStatus($bulkId);
                $data['bulk'] = [
                    'bulkId' => $bulkId,
                    'bulkStatus' => $this->formatBulkStatus($this->bulkStatus->getBulkStatus($bulkId)),
                    'operationCount' => $bulkDetail->getOperationCount(),
                    'operations' => array_map(
                        fn (OperationInterface $operation): array => [
                            'operationId' => $operation->getId(),
                            'status' => $this->formatOperationStatus((int)$operation->getStatus()),
                            'errorCode' => $operation->getErrorCode(),
                            'resultMessage' => $operation->getResultMessage(),
                            'resultSerializedData' => $operation->getResultSerializedData() ? json_decode($operation->getResultSerializedData()) : null,
                        ],
                        array_values($bulkDetail->getOperationsList())
                    ),
                ];
            }

            $pretty = json_encode($data, JSON_PRETTY_PRINT);
            $output->writeln("<info>$pretty</info>");

            return Cli::RETURN_SUCCESS;
        } catch (WebapiException $e) {
            $output->writeln("<error>{$e->getMessage()}</error>");
            return Cli::RETURN_FAILURE;
        } catch (NoSuchEntityException) {
            $output->writeln("Customer sync not found.");
            return Cli::RETURN_FAILURE;
        }

    }

    private function formatBulkStatus(int $bulkStatusId): string
    {
        $status = match ($bulkStatusId) {
            BulkSummaryInterface::NOT_STARTED => 'NOT_STARTED',
            BulkSummaryInterface::IN_PROGRESS => 'IN_PROGRESS',
            BulkSummaryInterface::FINISHED_SUCCESSFULLY => 'FINISHED_SUCCESSFULLY',
            BulkSummaryInterface::FINISHED_WITH_FAILURE => 'FINISHED_WITH_FAILURE',
            default => 'UNKNOWN'
        };

        return "$status ($bulkStatusId)";
    }

    private function formatOperationStatus(int $operationStatusId): string
    {
        $status = match ($operationStatusId) {
            OperationInterface::STATUS_TYPE_COMPLETE => 'COMPLETE',
            OperationInterface::STATUS_TYPE_RETRIABLY_FAILED => 'RETRIABLY_FAILED',
            OperationInterface::STATUS_TYPE_NOT_RETRIABLY_FAILED => 'NOT_RETRIABLY_FAILED',
            OperationInterface::STATUS_TYPE_OPEN => 'OPEN',
            OperationInterface::STATUS_TYPE_REJECTED => 'REJECTED',
            default => 'UNKNOWN'
        };

        return "$status ($operationStatusId)";
    }
}
