<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSCustomerSync\Support;

class Arrays {
    /**
     * Converts an associative array with keys in dot notation into a nested array.
     *
     * Usage:
     *
     *      $input = [
     *        "user.first_name" => "John",
     *        "user.last_name" => "Doe",
     *        "user.career.name" => "Programmer",
     *        "user.career.description" => "Develops stuff",
     *      ];
     *
     *      $output = [
     *        "user" => [
     *          "first_name" => "John",
     *          "last_name" => "Doe",
     *          "career" => [
     *            "Name" => "Programmer",
     *            "Description" => "Develops stuff"
     *          ]
     *        ]
     *      ]
     *
     *      $nestArray($input) == $output;
     *
     * @param array $dottedArray
     * @return array
     */
    public static function nestArray(array $dottedArray): array {
        $result = [];
        foreach ($dottedArray as $dottedKey => $value) {
            Arrays::setNested($result, $dottedKey, $value);
        }
        return $result;
    }

    /**
     * Assigns a value to a nested array given its path in dot notation.
     *
     * Usage:
     *
     *      $input = [];
     *
     *      $setNested($input, "foo.bar.baz", "hello world")
     *
     *      $input == [
     *        "foo" => [
     *          "bar" => [
     *            "baz" => "hello world",
     *          ],
     *        ],
     *      ];
     *
     * @param array $array array to assign the nested value to - taken by reference
     * @param string $dottedKey key in dot notation, e.g., "foo.bar.baz"
     * @param $value
     * @return void
     */
    public static function setNested(array &$array, string $dottedKey, $value): void {
        $keyParts = explode(".", $dottedKey);
        $currentLevel = &$array;
        for ($i = 0; $i < count($keyParts) - 1; ++$i) {
            $currentKey = &$keyParts[$i];
            if (!array_key_exists($currentKey, $currentLevel)) {
                $currentLevel[$currentKey] = [];
            }
            $currentLevel = &$currentLevel[$currentKey];
        }
        $currentLevel[$keyParts[count($keyParts) - 1]] = $value;
    }

    /**
     * Checks whether an array consists entirely of null values.
     *
     * @param array $array
     * @return bool true if array is empty or contains only NULL values
     */
    public static function allNull(array $array): bool {
        foreach ($array as $value) {
            if ($value !== null) {
                return false;
            }
        }
        return true;
    }
}
