<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSCustomerSync\Model;

use Magento\SaaSCustomerSync\Support\Arrays;
use Magento\AsynchronousOperations\Api\Data\OperationInterface;
use Magento\Framework\Event\ManagerInterface as EventManager;
use Magento\Framework\Serialize\SerializerInterface;
use Psr\Log\LoggerInterface;

class OperationConsumer
{
    /**
     * @var string
     */
    private const CUSTOMER_SYNCED_EVENT_NAME = 'saas_customer_synced';

    public function __construct(
        private readonly LoggerInterface        $logger,
        private readonly BulkAdapter            $bulkAdapter,
        private readonly SerializerInterface    $serializer,
        private readonly CustomerRepository     $customerRepository,
        private readonly EventManager           $eventManager,
        private readonly Config                 $config
    ) {
    }

    public function consume(OperationInterface $operation): void
    {
        $syncId = $operation->getBulkUuid();
        $operationId = $operation->getId();
        $ctx = ['syncId' => $syncId, 'operationId' => $operationId];
        $this->logger->info('Customer sync operation started.', $ctx);

        try {
            $this->process($operation);
            $this->bulkAdapter->completeOperation($operation);
        } catch (\Exception $e) {
            $ctx['exception'] = $e->getMessage();
            $this->logger->error('Failed to process customer sync operation.', $ctx);
            $this->bulkAdapter->failOperation(
                $operation,
                "Failed to process customer sync operation",
                [$e->getMessage()]
            );
            return;
        }

        $this->logger->info('Customer sync operation completed.', $ctx);
    }

    /**
     * @param OperationInterface $operation
     * @return void
     */
    private function process(OperationInterface $operation): void
    {
        $operationData = new OperationPayload($this->serializer->unserialize($operation->getSerializedData()));

        foreach ($this->customerRepository->findCustomerData(
            $operationData->getCustomerIds(),
            $this->config->getOperationConsumerBatchSize(),
        ) as $customers) {
            foreach ($customers as $customer) {
                $payload = $this->customerSyncedEventPayload($operation->getBulkUuid(), $customer);
                $this->eventManager->dispatch(self::CUSTOMER_SYNCED_EVENT_NAME, $payload);
            }
        }
    }

    /**
     * @param array $customerData
     * @param string $syncId
     * @return array
     */
    private function customerSyncedEventPayload(string $syncId, array $customerData): array {
        $payload = ["sync_id" => $syncId, "customer" => Arrays::nestArray($customerData)];

        if (Arrays::allNull($payload["customer"]["default_billing"])) {
            unset($payload["customer"]["default_billing"]);
        }

        if (Arrays::allNull($payload["customer"]["default_shipping"])) {
            unset($payload["customer"]["default_shipping"]);
        }

        return $payload;
    }
}
