<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSCustomerSync\Model;

use Exception;
use LogicException;
use Magento\AsynchronousOperations\Api\BulkStatusInterface;
use Magento\AsynchronousOperations\Api\Data\OperationInterfaceFactory;
use Magento\AsynchronousOperations\Model\OperationManagement;
use Magento\AsynchronousOperations\Api\SaveMultipleOperationsInterface;
use Magento\Framework\Bulk\BulkManagementInterface;
use Magento\Framework\Bulk\OperationInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\MessageQueue\ConnectionTypeResolver;
use Magento\Framework\Serialize\SerializerInterface;
use Psr\Log\LoggerInterface;

class BulkAdapter
{

    public function __construct(
        private readonly LoggerInterface                 $logger,
        private readonly SerializerInterface             $serializer,
        private readonly BulkManagementInterface         $bulkManagement,
        private readonly BulkStatusInterface             $bulkStatus,
        private readonly OperationInterfaceFactory       $operationFactory,
        private readonly SaveMultipleOperationsInterface $saveMultipleOperations,
        private readonly ConnectionTypeResolver          $connectionTypeResolver,
        private readonly OperationManagement             $operationManagement,
    ) {
    }

    public function isAMQPAvailable(): bool
    {
        try {
            // amqp connection is defined in etc/queue_topology.xml
            $this->connectionTypeResolver->getConnectionType('amqp');
            return true;
        } catch (LogicException) {
            return false;
        }
    }

    public function scheduleOperation(string $syncId, int $operationId, OperationPayload $operationPayload): bool
    {
        $operation = $this->operationFactory->create([
            'data' => [
                'operation_key' => $operationId,
                'bulk_uuid' => $syncId,
                'topic_name' => 'saas.customer.sync.bulk',
                'serialized_data' => $this->serializer->serialize($operationPayload->getCustomerIds()),
                'status' => OperationInterface::STATUS_TYPE_OPEN,
            ],
        ]);

        $scheduled = $this->bulkManagement->scheduleBulk($syncId, [$operation], "SaaS customer sync $syncId");
        if (!$scheduled) {
            $this->logger->error("Failed to schedule bulk operation. Check logs.", ['syncId' => $syncId, 'operationId' => $operationId]);
            return false;
        }

        try {
            $this->saveMultipleOperations->execute([$operation]);
            return true;
        } catch (Exception $e) {
            $this->logger->error("Failed to save bulk operation.", ['syncId' => $syncId, 'operationId' => $operationId, 'exception' => $e]);
            return false;
        }
    }

    public function failOperation(OperationInterface $operation, string $message, ?array $error = []): void
    {
        $this->operationManagement->changeOperationStatus(
            $operation->getBulkUuid(),
            $operation->getId(),
            OperationInterface::STATUS_TYPE_NOT_RETRIABLY_FAILED,
            message: __($message),
            data: $operation->getSerializedData(),
            resultData: $error ? $this->serializer->serialize($error) : null
        );
    }

    public function completeOperation(OperationInterface $operation): void
    {
        $this->operationManagement->changeOperationStatus(
            $operation->getBulkUuid(),
            $operation->getId(),
            OperationInterface::STATUS_TYPE_COMPLETE,
        );
    }

    /**
     * @throws NoSuchEntityException
     */
    public function findOperations(string $syncId): array
    {
        $operationList = $this->bulkStatus->getBulkShortStatus($syncId)->getOperationsList();

        return array_map(function ($operation) {
            return [
                'operationId' => $operation->getId(),
                'status' => $operation->getStatus()
            ];
        }, $operationList);
    }
}
