<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSCustomerSync\Console\Command;

use Magento\Framework\Console\Cli;
use Magento\Framework\Webapi\Exception as WebapiException;
use Magento\SaaSCustomerSync\Api\CustomerSyncManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

class StartCustomerSync extends Command
{
    private const COMMAND_NAME = 'saas:customer-sync:start';

    public function __construct(private readonly CustomerSyncManagerInterface $customerSyncManager)
    {
        parent::__construct();
    }


    protected function configure(): void
    {
        $this
            ->setName(self::COMMAND_NAME)
            ->setDescription('Synchronize customers with SaaS within a certain time frame.')
            ->addArgument(
                'created_from',
                InputArgument::REQUIRED,
                'Created from date time in ISO-8601 format.'
            )
            ->addArgument(
                'created_to',
                InputArgument::REQUIRED,
                'Created to date time in ISO-8601 format.'
            );
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $createdFrom = $input->getArgument('created_from');
        $createdTo = $input->getArgument('created_to');

        try {
            $response = $this->customerSyncManager->createCustomerSync($createdFrom, $createdTo);
            $data = [
                'sync_id' => $response->getSyncId(),
                'created_from' => $response->getCreatedFrom(),
                'created_to' => $response->getCreatedTo(),
                'customer_count' => $response->getCustomerCount()
            ];

            $pretty = json_encode($data, JSON_PRETTY_PRINT);
            $output->writeln("<info>$pretty</info>");

            return Cli::RETURN_SUCCESS;
        } catch (WebapiException $e) {
            $output->writeln("<error>{$e->getMessage()}</error>");
            return Cli::RETURN_FAILURE;
        }
    }
}
