<?php
/************************************************************************
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSCustomerSync\Test\Unit\Model;

use Generator;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\SaaSCustomerSync\Event\ScopeMetadataCollector;
use Magento\SaaSCustomerSync\Model\BulkAdapter;
use Magento\SaaSCustomerSync\Model\Config;
use Magento\SaaSCustomerSync\Model\CustomerRepository;
use Magento\SaaSCustomerSync\Model\OperationConsumer;
use Magento\AsynchronousOperations\Model\Operation;
use Magento\Framework\Bulk\OperationInterface;
use Magento\AsynchronousOperations\Model\OperationStatusValidator;
use Magento\SaaSCustomerSync\SaaS\SaaSConfig;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Log\NullLogger;
use Magento\Framework\Event\ManagerInterface as EventManager;
use function PHPUnit\Framework\anything;
use function PHPUnit\Framework\exactly;
use function PHPUnit\Framework\once;
use function PHPUnit\Framework\never;

class OperationConsumerTest extends TestCase
{
    /**
     * @var BulkAdapter&MockObject
     */
    private BulkAdapter $bulkAdapterMock;

    /**
     * @var CustomerRepository&MockObject
     */
    private CustomerRepository $customerRepoMock;

    /**
     * @var EventManager&MockObject
     */
    private EventManager $eventManager;

    /**
     * @var OperationConsumer
     */
    private OperationConsumer $operationConsumer;

    /**
     * @var ScopeMetadataCollector&MockObject
     */
    private ScopeMetadataCollector $scopeMetadataCollector;

    /**
     * @var SaaSConfig&MockObject
     */
    private SaaSConfig $saaSConfig;

    /**
     * @var Json
     */
    private Json $serializer;

    public function setUp(): void
    {
        $this->bulkAdapterMock = $this->createMock(BulkAdapter::class);
        $this->customerRepoMock = $this->createMock(CustomerRepository::class);
        $this->eventManager = $this->createMock(EventManager::class);
        $this->scopeMetadataCollector = $this->createMock(ScopeMetadataCollector::class);
        $this->saaSConfig = $this->createMock(SaaSConfig::class);

        $config = new Config();
        $config->setOperationConsumerBatchSize(2);

        $this->serializer = new Json();

        $this->operationConsumer = new OperationConsumer(
            new NullLogger(),
            $this->bulkAdapterMock,
            $this->serializer,
            $this->customerRepoMock,
            $this->eventManager,
            $config,
            $this->scopeMetadataCollector,
            $this->saaSConfig
        );
    }

    public function test_shouldEmitSaasCustomerSynced_whenConsumingOperation(): void
    {
        $operation = $this->getOperation([1, 2, 3]);
        $scope = [
            'websiteCode' => 'foo',
            'storeCode' => 'bar',
            'storeViewCode' => 'baz',
        ];

        $merchantInfo = [
            'mageId'        => 'paco',
            'environmentId' => 'pepe'
        ];

        $customer1 = [
            'entity_id' => 1,
            'created_at' => '2022-01-01 00:00:00',
            'updated_at' => '2022-01-01 00:00:00',
            'email' => 'c1@example.com',
            'firstname' => 'Customer',
            'lastname' => 'One',
            'dob' => '2022-01-01 00:00:00',
        ];

        $customer2 = [
            'entity_id' => 2,
            'created_at' => '2022-02-02 00:00:00',
            'updated_at' => '2022-02-02 00:00:00',
            'email' => 'c2@example.com',
            'firstname' => 'Customer',
            'lastname' => 'Two',
            'dob' => '2022-02-02 00:00:00',
        ];

        $customer3 = [
            'entity_id' => 3,
            'created_at' => '2022-03-03 00:00:00',
            'updated_at' => '2022-03-03 00:00:00',
            'email' => 'c3@example.com',
            'firstname' => 'Customer',
            'lastname' => 'Three',
            'dob' => '2022-03-03 00:00:00',
        ];

        $this->customerRepoMock->expects(once())
            ->method('findCustomerData')
            ->with([1, 2, 3], 2)
            ->willReturn($this->generator([$customer1, $customer2, $customer3]));

        $this->scopeMetadataCollector->expects(once())
            ->method('get')
            ->willReturn($scope);

        $this->saaSConfig->expects(once())
            ->method('getMageId')
            ->willReturn($merchantInfo['mageId']);

        $this->saaSConfig->expects(once())
            ->method('getEnvironmentId')
            ->willReturn($merchantInfo['environmentId']);

        $this->eventManager->expects(exactly(3))
            ->method('dispatch')
            ->withConsecutive(
                ['saas_customer_synced', 'object' => $this->saasCustomerSyncEvent(
                    $customer1,
                    "1",
                    $scope,
                    $merchantInfo
                )
                ],
                ['saas_customer_synced', 'object' => $this->saasCustomerSyncEvent(
                    $customer2,
                    "1",
                    $scope,
                    $merchantInfo
                )
                ],
                ['saas_customer_synced', 'object' => $this->saasCustomerSyncEvent(
                    $customer3,
                    "1",
                    $scope,
                    $merchantInfo
                )
                ]
            );

        $this->bulkAdapterMock->expects(once())
            ->method('completeOperation')
            ->with($operation);

        $this->bulkAdapterMock->expects(never())
            ->method('failOperation');

        $this->operationConsumer->consume($operation);
    }

    public function test_shouldMarkOperationAsFailed_whenCustomerRepoThrowsException(): void
    {
        $operation = $this->getOperation([1, 2, 3]);

        $this->customerRepoMock->expects(once())
            ->method('findCustomerData')
            ->with([1, 2, 3], 2)
            ->willThrowException(new \Exception('Something went wrong'));

        $this->eventManager->expects(never())
            ->method('dispatch');

        $this->bulkAdapterMock->expects(never())
            ->method('completeOperation');

        $this->bulkAdapterMock->expects(once())
            ->method('failOperation')
            ->with(
                $operation,
                anything(),
                ['Something went wrong']
            );

        $this->operationConsumer->consume($operation);
    }

    /**
     * @param  $customerIds
     * @return Operation
     */
    public function getOperation(array $customerIds): Operation
    {
        return new Operation(
            $this->createMock(OperationStatusValidator::class),
            [
                OperationInterface::ID => 1,
                OperationInterface::BULK_ID => "1",
                OperationInterface::SERIALIZED_DATA => $this->serializer->serialize($customerIds)
            ]
        );
    }

    private function generator(mixed...$values): Generator
    {
        yield from $values;
    }

    private function saasCustomerSyncEvent(
        array $customerData,
        string $syncId,
        array $scope,
        array $merchantInfo
    ): array {
        return [
            "merchant"      => $merchantInfo,
            "source"        => ['type' => 'ADOBE_COMMERCE', 'customer_source_id' => $customerData['entity_id']],
            "syncId"        => $syncId,
            "customerId"    => $customerData['entity_id'],
            "createdAt"     => $customerData['created_at'],
            "updatedAt"     => $customerData['updated_at'],
            "email"         => $customerData['email'],
            "firstName"     => $customerData['firstname'],
            "lastName"      => $customerData['lastname'],
            "dateOfBirth"   => $customerData['dob'],
            "scope"         => $scope,
        ];
    }
}
