<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSCustomerSync\Test\Unit\Model;

use Exception;
use Generator;
use Magento\Framework\Bulk\BulkSummaryInterface;
use Magento\Framework\Bulk\OperationInterface;
use Magento\Framework\Webapi\Exception as WebapiException;
use Magento\SaaSCustomerSync\Api\CustomerSyncManagerInterface;
use Magento\SaaSCustomerSync\Model\BulkAdapter;
use Magento\SaaSCustomerSync\Model\Config;
use Magento\SaaSCustomerSync\Model\CustomerRepository;
use Magento\SaaSCustomerSync\Model\CustomerSyncManager;
use Magento\SaaSCustomerSync\Model\OperationPayload;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Log\NullLogger;
use function PHPUnit\Framework\anything;
use function PHPUnit\Framework\assertNotNull;
use function PHPUnit\Framework\assertThat;
use function PHPUnit\Framework\equalTo;
use function PHPUnit\Framework\exactly;
use function PHPUnit\Framework\isInstanceOf;
use function PHPUnit\Framework\once;

class CustomerSyncManagerTest extends TestCase
{
    private FixedIdentityGenerator $identityGenerator;
    private BulkAdapter&MockObject $bulkAdapterMock;
    private CustomerRepository&MockObject $customerRepoMock;
    private Config $config;

    private CustomerSyncManagerInterface $customerSyncManager;

    public function setUp(): void
    {
        $this->identityGenerator = new FixedIdentityGenerator();
        $this->bulkAdapterMock = $this->createMock(BulkAdapter::class);
        $this->customerRepoMock = $this->createMock(CustomerRepository::class);

        $this->config = new Config();
        $this->config->setOperationBatchSize(2);

        $this->customerSyncManager = new CustomerSyncManager(
            $this->identityGenerator,
            new NullLogger(),
            $this->customerRepoMock,
            $this->bulkAdapterMock,
            $this->config
        );
    }

    public function test_shouldCreateSync_forTheGivenTimeframe_whenCustomersInTheTimeframe(): void
    {
        $syncId = $this->identityGenerator->getId();

        $this->bulkAdapterMock->expects(once())
            ->method('isAMQPAvailable')
            ->willReturn(true);

        $this->customerRepoMock->expects(once())
            ->method('findCustomerIds')
            ->with(anything(), anything(), $this->config->getOperationBatchSize())
            ->willReturn($this->generator([1, 2], [3, 4], [5]));

        $this->bulkAdapterMock->expects(exactly(3))
            ->method('scheduleOperation')
            ->withConsecutive(
                [$syncId, 1, new OperationPayload([1, 2])],
                [$syncId, 2, new OperationPayload([3, 4])],
                [$syncId, 3, new OperationPayload([5])],
            )
            ->willReturn(true);

        $createdFrom = '2022-01-01 00:00:00';
        $createdTo = '2023-01-01 00:00:00';

        $customerSync = $this->customerSyncManager->createCustomerSync($createdFrom, $createdTo);

        assertNotNull($customerSync);
        assertThat($customerSync->getSyncId(), equalTo($syncId));
        assertThat($customerSync->getCreatedFrom(), equalTo($createdFrom));
        assertThat($customerSync->getCreatedTo(), equalTo($createdTo));
        assertThat($customerSync->getCustomerCount(), equalTo(5));
    }

    public function test_shouldCreateSync_forTheGivenTimeframe_whenEmptyCustomersInTheTimeframe(): void
    {
        $syncId = $this->identityGenerator->getId();

        $this->bulkAdapterMock->expects(once())
            ->method('isAMQPAvailable')
            ->willReturn(true);

        $createdFrom = '2022-01-01 00:00:00';
        $createdTo = '2023-01-01 00:00:00';

        $customerSync = $this->customerSyncManager->createCustomerSync($createdFrom, $createdTo);
        assertNotNull($customerSync);
        assertThat($customerSync->getSyncId(), equalTo($syncId));
        assertThat($customerSync->getCreatedFrom(), equalTo($createdFrom));
        assertThat($customerSync->getCreatedTo(), equalTo($createdTo));
        assertThat($customerSync->getCustomerCount(), equalTo(0));
    }

    public function test_shouldFailWith500_whenSchedulingBulkFails(): void
    {
        $syncId = $this->identityGenerator->getId();

        $this->bulkAdapterMock->expects(once())
            ->method('isAMQPAvailable')
            ->willReturn(true);

        $this->customerRepoMock->expects(once())
            ->method('findCustomerIds')
            ->willReturn($this->generator([1]));

        $this->bulkAdapterMock->expects(once())
            ->method('scheduleOperation')
            ->with($syncId, 1, new OperationPayload([1]))
            ->willReturn(false);

        $createdFrom = '2022-01-01 00:00:00';
        $createdTo = '2023-01-01 00:00:00';

        $this->assertThrowsWebapiException(
            fn () => $this->customerSyncManager->createCustomerSync($createdFrom, $createdTo),
            WebapiException::HTTP_INTERNAL_ERROR,
        );
    }

    public function test_shouldFailWith500_whenAMQPIsNotAvailable(): void
    {
        $this->bulkAdapterMock->expects(once())
            ->method('isAMQPAvailable')
            ->willReturn(false);

        $createdFrom = '2022-01-01 00:00:00';
        $createdTo = '2023-01-01 00:00:00';

        $this->assertThrowsWebapiException(
            fn () => $this->customerSyncManager->createCustomerSync($createdFrom, $createdTo),
            WebapiException::HTTP_INTERNAL_ERROR,
        );
    }

    public function test_shouldRejectSync_whenCreatedFromIsNotDateTime(): void
    {
        $createdFrom = 'boom';
        $createdTo = '2023-01-01 00:00:00';

        $this->assertThrowsWebapiException(
            fn () => $this->customerSyncManager->createCustomerSync($createdFrom, $createdTo),
            WebapiException::HTTP_BAD_REQUEST,
        );
    }

    public function test_shouldRejectSync_whenCreatedToIsNotDateTime(): void
    {
        $createdFrom = '2022-01-01 00:00:00';
        $createdTo = 'boom';

        $this->assertThrowsWebapiException(
            fn () => $this->customerSyncManager->createCustomerSync($createdFrom, $createdTo),
            WebapiException::HTTP_BAD_REQUEST,
        );
    }

    public function test_shouldRejectSync_whenCreatedFromIsGreaterThanCreatedTo(): void
    {
        $createdFrom = '2022-01-02 00:00:00';
        $createdTo = '2022-01-01 00:00:00';

        $this->assertThrowsWebapiException(
            fn () => $this->customerSyncManager->createCustomerSync($createdFrom, $createdTo),
            WebapiException::HTTP_BAD_REQUEST,
        );
    }

    public function test_shouldRejectSync_whenCreatedFromIsEqualToCreatedTo(): void
    {
        $createdFrom = '2022-01-01 00:00:00';
        $createdTo = '2022-01-01 00:00:00';

        $this->assertThrowsWebapiException(
            fn () => $this->customerSyncManager->createCustomerSync($createdFrom, $createdTo),
            WebapiException::HTTP_BAD_REQUEST,
        );
    }

    /**
     * @dataProvider syncStatusTestCases
     */
    public function test_syncStatus(array $testCase): void
    {
        [
            'operationStatus' => $operationStatus,
            'syncStatus' => $syncStatus
        ] = $testCase;

        $syncId = $this->identityGenerator->getId();

        $operations = array_map(
            fn ($idx, $status) => [
                'operationId' => $idx,
                'status' => $status,
            ],
            array_keys($operationStatus), $operationStatus);

        $this->bulkAdapterMock->expects(once())
            ->method('findOperations')
            ->with($syncId)
            ->willReturn($operations);

        $customerSync = $this->customerSyncManager->getCustomerSync($syncId);

        assertNotNull($customerSync);
        assertThat($customerSync->getSyncId(), equalTo($syncId));
        assertThat($customerSync->getStatus(), equalTo($syncStatus));
    }


    private function syncStatusTestCases(): array
    {
        return [
            [[
                'operationStatus' => [],
                'syncStatus' => BulkSummaryInterface::NOT_STARTED
            ]],
            [[
                'operationStatus' => [OperationInterface::STATUS_TYPE_OPEN, OperationInterface::STATUS_TYPE_OPEN],
                'syncStatus' => BulkSummaryInterface::NOT_STARTED
            ]],
            [[
                'operationStatus' => [OperationInterface::STATUS_TYPE_OPEN, OperationInterface::STATUS_TYPE_COMPLETE],
                'syncStatus' => BulkSummaryInterface::IN_PROGRESS
            ]],
            [[
                'operationStatus' => [OperationInterface::STATUS_TYPE_OPEN, OperationInterface::STATUS_TYPE_NOT_RETRIABLY_FAILED],
                'syncStatus' => BulkSummaryInterface::IN_PROGRESS
            ]],
            [[
                'operationStatus' => [OperationInterface::STATUS_TYPE_COMPLETE, OperationInterface::STATUS_TYPE_COMPLETE],
                'syncStatus' => BulkSummaryInterface::FINISHED_SUCCESSFULLY
            ]],
            [[
                'operationStatus' => [OperationInterface::STATUS_TYPE_COMPLETE, OperationInterface::STATUS_TYPE_NOT_RETRIABLY_FAILED],
                'syncStatus' => BulkSummaryInterface::FINISHED_WITH_FAILURE
            ]],
            [[
                'operationStatus' => [OperationInterface::STATUS_TYPE_NOT_RETRIABLY_FAILED, OperationInterface::STATUS_TYPE_NOT_RETRIABLY_FAILED],
                'syncStatus' => BulkSummaryInterface::FINISHED_WITH_FAILURE
            ]],
        ];
    }

    private function generator(mixed...$values): Generator
    {
        yield from $values;
    }

    private function assertThrowsWebapiException(callable $callable, int $httpStatusCode): void
    {
        $this->assertThrowsException(
            $callable,
            WebapiException::class,
            fn (WebapiException $e) => assertThat($e->getHttpCode(), equalTo($httpStatusCode))
        );
    }

    private function assertThrowsException(callable $callable, string $exception, callable $exceptionCallable = null): void
    {
        try {
            $callable();
        } catch (Exception $e) {
            assertThat($e, isInstanceOf($exception));
            if ($exceptionCallable != null) {
                $exceptionCallable($e);
            }
        }
    }
}
