<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSCustomerSync\Test\Unit\Model;

use Generator;
use Magento\SaaSCustomerSync\Model\BulkAdapter;
use Magento\SaaSCustomerSync\Model\Config;
use Magento\SaaSCustomerSync\Model\CustomerRepository;
use Magento\SaaSCustomerSync\Model\OperationConsumer;
use Magento\AsynchronousOperations\Model\Operation;
use Magento\Framework\Bulk\OperationInterface;
use Magento\AsynchronousOperations\Model\OperationStatusValidator;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Log\NullLogger;
use Magento\Framework\Serialize\Serializer\Serialize;
use Magento\Framework\Event\ManagerInterface as EventManager;
use function PHPUnit\Framework\anything;
use function PHPUnit\Framework\exactly;
use function PHPUnit\Framework\once;
use function PHPUnit\Framework\never;

class OperationConsumerTest extends TestCase
{
    /**
     * @var BulkAdapter&MockObject
     */
    private BulkAdapter $bulkAdapterMock;

    /**
     * @var CustomerRepository&MockObject
     */
    private CustomerRepository $customerRepoMock;

    /**
     * @var EventManager&MockObject
     */
    private EventManager $eventManager;

    /**
     * @var OperationConsumer
     */
    private OperationConsumer $operationConsumer;

    public function setUp(): void
    {
        $this->bulkAdapterMock = $this->createMock(BulkAdapter::class);
        $this->customerRepoMock = $this->createMock(CustomerRepository::class);
        $this->eventManager = $this->createMock(EventManager::class);

        $config = new Config();
        $config->setOperationConsumerBatchSize(2);

        $this->operationConsumer = new OperationConsumer(
            new NullLogger(),
            $this->bulkAdapterMock,
            new Serialize(),
            $this->customerRepoMock,
            $this->eventManager,
            $config
        );
    }

    public function test_shouldEmitSaasCustomerSynced_whenConsumingOperation(): void
    {
        $operation = $this->getOperation([1, 2, 3]);

        $customer1 = [
            'entity_id' => 1,
            'created_at' => '2022-01-01 00:00:00',
            'updated_at' => '2022-01-01 00:00:00',
            'email' => 'c1@example.com',
            'firstname' => 'Customer',
            'lastname' => 'One',
            'dob' => '2022-01-01 00:00:00',
        ];

        $customer2 = [
            'entity_id' => 2,
            'created_at' => '2022-02-02 00:00:00',
            'updated_at' => '2022-02-02 00:00:00',
            'email' => 'c2@example.com',
            'firstname' => 'Customer',
            'lastname' => 'Two',
            'dob' => '2022-02-02 00:00:00',
        ];

        $customer3 = [
            'entity_id' => 3,
            'created_at' => '2022-03-03 00:00:00',
            'updated_at' => '2022-03-03 00:00:00',
            'email' => 'c3@example.com',
            'firstname' => 'Customer',
            'lastname' => 'Three',
            'dob' => '2022-03-03 00:00:00',
        ];

        $this->customerRepoMock->expects(once())
            ->method('findCustomerData')
            ->with([1, 2, 3], 2)
            ->willReturn($this->generator([$customer1, $customer2, $customer3]));

        $this->eventManager->expects(exactly(3))
            ->method('dispatch')
            ->withConsecutive(
                ['saas_customer_synced', 'object' => $this->saasCustomerSyncEvent($customer1, "1")],
                ['saas_customer_synced', 'object' => $this->saasCustomerSyncEvent($customer2, "1")],
                ['saas_customer_synced', 'object' => $this->saasCustomerSyncEvent($customer3, "1")]
            );

        $this->bulkAdapterMock->expects(once())
            ->method('completeOperation')
            ->with($operation);

        $this->bulkAdapterMock->expects(never())
            ->method('failOperation');

        $this->operationConsumer->consume($operation);
    }

    public function test_shouldMarkOperationAsFailed_whenCustomerRepoThrowsException(): void
    {
        $operation = $this->getOperation([1, 2, 3]);

        $this->customerRepoMock->expects(once())
            ->method('findCustomerData')
            ->with([1, 2, 3], 2)
            ->willThrowException(new \Exception('Something went wrong'));

        $this->eventManager->expects(never())
            ->method('dispatch');

        $this->bulkAdapterMock->expects(never())
            ->method('completeOperation');

        $this->bulkAdapterMock->expects(once())
            ->method('failOperation')
            ->with(
                $operation,
                anything(),
                ['Something went wrong']
            );

        $this->operationConsumer->consume($operation);
    }

    /**
     * @param $customerIds
     * @return Operation
     */
    public function getOperation(array $customerIds): Operation
    {
        return new Operation(
            $this->createMock(OperationStatusValidator::class),
            [
                OperationInterface::ID => 1,
                OperationInterface::BULK_ID => "1",
                OperationInterface::SERIALIZED_DATA => serialize($customerIds)
            ]
        );
    }

    private function generator(mixed...$values): Generator
    {
        yield from $values;
    }

    private function saasCustomerSyncEvent(array $customerData, string $syncId): array
    {
        return [
            "syncId"        => $syncId,
            "customerId"    => $customerData['entity_id'],
            "createdAt"     => $customerData['created_at'],
            "updatedAt"     => $customerData['updated_at'],
            "email"         => $customerData['email'],
            "firstName"     => $customerData['firstname'],
            "lastName"      => $customerData['lastname'],
            "dateOfBirth"   => $customerData['dob'],
        ];
    }
}
