<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSCustomerSync\Model;

use DateTime;
use Exception;
use Magento\Framework\Bulk\BulkSummaryInterface;
use Magento\Framework\Bulk\OperationInterface;
use Magento\Framework\DataObject\IdentityGeneratorInterface;
use Magento\Framework\Webapi\Exception as WebapiException;
use Magento\SaaSCustomerSync\Api\CustomerSyncManagerInterface;
use Magento\SaaSCustomerSync\Model\Data\CreateCustomerSyncResponse;
use Magento\SaaSCustomerSync\Model\Data\GetCustomerSyncResponse;
use Psr\Log\LoggerInterface;

class CustomerSyncManager implements CustomerSyncManagerInterface
{

    public function __construct(
        private readonly IdentityGeneratorInterface $identityGenerator,
        private readonly LoggerInterface            $logger,
        private readonly CustomerRepository         $customerRepository,
        private readonly BulkAdapter                $bulkAdapter,
        private readonly Config                     $config,
    ) {
    }

    public function createCustomerSync(string $created_from, string $created_to): CreateCustomerSyncResponse
    {
        [$createdFromDateTime, $createdToDateTime] = $this->parseTimeframe($created_from, $created_to);
        if (!$this->bulkAdapter->isAMQPAvailable()) {
            $this->throwInternalServerError(
                'Can\'t start Customer Sync due to missing AMQP connection.'
                . ' Request to the administrator of the Adobe Commerce instance to configure Message Queue Framework accordingly.'
                . ' See: https://experienceleague.adobe.com/docs/commerce-operations/configuration-guide/message-queues/message-queue-framework.html?lang=en'
            );
        }

        $syncId = $this->identityGenerator->generateId();

        $operationCount = 0;
        $customerCount = 0;
        foreach ($this->customerRepository->findCustomerIds(
            $createdFromDateTime,
            $createdToDateTime,
            $this->config->getOperationBatchSize(),
        ) as $customerIds) {

            if (!empty($customerIds)) {
                $operationId = ++$operationCount;
                $customerCount += count($customerIds);

                $scheduled = $this->bulkAdapter->scheduleOperation(
                    $syncId,
                    $operationId,
                    new OperationPayload($customerIds)
                );

                if ($scheduled) {
                    $this->logger->debug(
                        "Added operation to customer sync bulk.",
                        ['syncId' => $syncId, 'operationId' => $operationId],
                    );
                } else {
                    $this->throwInternalServerError("Failed to schedule operation $operationId for sync $syncId");
                }
            }
        }

        $ctx = [
            'syncId' => $syncId,
            'createdFrom' => $createdFromDateTime,
            'createdTo' => $createdToDateTime,
            'operationCount' => $operationCount,
            'customerCount' => $customerCount,
        ];
        if ($customerCount == 0) {
            $this->logger->info("Customer sync not scheduled, customers not found in the given timeframe.", $ctx);
        } else {
            $this->logger->info("Scheduled customer sync.", $ctx);
        }

        return new CreateCustomerSyncResponse(
            $syncId,
            $created_from,
            $created_to,
            $customerCount,
        );
    }

    public function getCustomerSync(string $sync_id): GetCustomerSyncResponse
    {
        return new GetCustomerSyncResponse(
            $sync_id,
            $this->getBulkStatus($sync_id),
        );
    }

    /**
     * When magento2 bug is fixed, it could be replaced by calling
     * {@see \Magento\Framework\Bulk\BulkStatusInterface::getBulkStatus}
     * @param string $bulkUuid
     * @return int
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     * @see https://github.com/magento/magento2/issues/36911">magento/magento2#36911
     */
    private function getBulkStatus(string $bulkUuid): int
    {
        $operations = $this->bulkAdapter->findOperations($bulkUuid);
        if (empty($operations)) {
            return BulkSummaryInterface::NOT_STARTED;
        }

        $operationsCount = count($operations);
        $statusCounts = array_count_values(array_column($operations, 'status'));

        $openCount = $statusCounts[OperationInterface::STATUS_TYPE_OPEN] ?? 0;
        if ($openCount == $operationsCount) {
            return BulkSummaryInterface::NOT_STARTED;
        }
        if ($openCount > 0) {
            return BulkSummaryInterface::IN_PROGRESS;
        }

        $completedCount = $statusCounts[OperationInterface::STATUS_TYPE_COMPLETE] ?? 0;
        if ($completedCount == $operationsCount) {
            return BulkSummaryInterface::FINISHED_SUCCESSFULLY;
        }

        return BulkSummaryInterface::FINISHED_WITH_FAILURE;
    }

    private function parseTimeframe(string $created_from, string $created_to): array
    {
        $createdFromDateTime = $this->parseDateTime($created_from)
            ?: $this->throwBadRequest('created_from can\'t be parsed in ISO8601 format.');
        $createdToDateTime = $this->parseDateTime($created_to)
            ?: $this->throwBadRequest('created_to can\'t be parsed in ISO8601 format.');

        if ($createdFromDateTime >= $createdToDateTime) {
            $this->throwBadRequest('created_to should be greater than created_from.');
        }

        return [
            $createdFromDateTime,
            $createdToDateTime,
        ];
    }

    private function parseDateTime(string $dateTime): DateTime|bool
    {
        try {
            return new DateTime($dateTime);
        } catch (Exception) {
            return false;
        }
    }

    private function throwInternalServerError($msg)
    {
        throw new WebapiException(
            __($msg),
            httpCode: WebapiException::HTTP_INTERNAL_ERROR,
        );
    }

    private function throwBadRequest(string $msg)
    {
        throw new WebapiException(
            __($msg),
            httpCode: WebapiException::HTTP_BAD_REQUEST,
        );
    }
}
