<?php
/************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2023 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 * ************************************************************************
 */
declare(strict_types=1);

namespace Magento\SaaSCustomerSync\Model;

use DateTime;
use Generator;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\DB\Query\BatchIteratorFactory;

class CustomerRepository
{

    public function __construct(
        private readonly ResourceConnection $resourceConnection,
        private readonly BatchIteratorFactory $batchIteratorFactory,
    ) {
    }

    public function findCustomerIds(DateTime $createdFrom, DateTime $createdTo, int $batchSize): Generator
    {
        $connection = $this->resourceConnection->getConnection();
        $tableName = $this->resourceConnection->getTableName('customer_entity');

        $select = $connection->select()
            ->from($tableName, ['entity_id'])
            ->where('created_at >= ?', $createdFrom)
            ->where('created_at <= ?', $createdTo);

        $iterator = $this->batchIteratorFactory->create([
            'select' => $select,
            'batchSize' => $batchSize,
            'correlationName' => $tableName,
            'rangeField' => 'entity_id',
            'rangeFieldAlias' => 'entity_id',
        ]);

        foreach ($iterator as $batchSelect) {
            yield $connection->fetchCol($batchSelect);
        }
    }

    public function findCustomerData(array $customerIds, int $batchSize): Generator
    {
        $connection = $this->resourceConnection->getConnection();
        $tableName = $this->resourceConnection->getTableName('customer_entity');

        $entityIds = implode(", ", $customerIds);
        $select = $connection->select()
            ->from($tableName, [
                'entity_id', 'email', 'firstname', 'lastname', 'dob', 'created_at', 'updated_at'
            ])
            ->where("entity_id in ($entityIds)");

        $iterator = $this->batchIteratorFactory->create([
            'select' => $select,
            'batchSize' => $batchSize,
            'correlationName' => $tableName,
            'rangeField' => 'entity_id',
            'rangeFieldAlias' => 'entity_id',
        ]);

        foreach ($iterator as $batchSelect) {
            yield $connection->fetchAll($batchSelect);
        }
    }
}
