<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\SaaSCommon\Console\Command;

use Magento\DataExporter\Model\Indexer\ConfigOptionsHandler;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\Console\Cli;
use Magento\SaaSCommon\Console\ProgressBarManager;
use Magento\SaaSCommon\Model\ResyncOptions;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Magento\SaaSCommon\Model\ResyncManager;
use Magento\SaaSCommon\Model\ResyncManagerPool;

/**
 * CLI command for Saas feed data resync
 */
class Resync extends Command
{
    /**
     * @var ResyncManagerPool
     */
    private $resyncManagerPool;

    /**
     * @var ResyncManager
     */
    private $resyncManager;

    /**
     * @var string
     */
    private $feedNames = [];

    /**
     * @var ResyncOptions
     */
    private ResyncOptions $resyncOptions;

    /**
     * @var ProgressBarManager
     */
    private ?ProgressBarManager $progressBarManager;

    /**
     * @var ConfigOptionsHandler
     */
    private ConfigOptionsHandler $configOptionsHandler;

    /**
     * @param ResyncManagerPool $resyncManagerPool
     * @param ResyncOptions $resyncOptions
     * @param string $name
     * @param array $feedNames
     * @param ProgressBarManager|null $progressBarManager
     * @param ConfigOptionsHandler|null $configOptionsHandler
     */
    public function __construct(
        ResyncManagerPool $resyncManagerPool,
        ResyncOptions $resyncOptions,
        $name = '',
        $feedNames = [],
        ?ProgressBarManager $progressBarManager = null,
        ?ConfigOptionsHandler $configOptionsHandler = null
    ) {
        $this->resyncOptions = $resyncOptions;
        $this->resyncManagerPool = $resyncManagerPool;
        $this->feedNames = $feedNames;
        $this->progressBarManager = $progressBarManager ?? ObjectManager::getInstance()->get(ProgressBarManager::class);
        $this->configOptionsHandler = $configOptionsHandler
            ?? ObjectManager::getInstance()->get(ConfigOptionsHandler::class);
        parent::__construct($name);
    }

    /**
     * @inheritDoc
     */
    protected function configure()
    {
        $this->setDescription('Re-syncs feed data to SaaS service.');
        $this->addOption(
            ResyncOptions::FEED_OPTION,
            null,
            InputOption::VALUE_REQUIRED,
            'Feed name to fully re-sync to SaaS service. Available feeds: ' . implode(', ', $this->feedNames)
        );

        foreach ($this->resyncOptions->getOptionsList() as $optionConfig) {
            $this->addOption(
                $optionConfig['name'],
                null,
                $optionConfig['mode'] ?? null,
                $optionConfig['description'] ?? null
            );
        }

        parent::configure();
    }

    /**
     * Execute the command to re-sync SaaS data
     *
     * @param InputInterface $input
     * @param OutputInterface $output
     * @return int|void
     */
    protected function execute(InputInterface $input, OutputInterface $output): ?int
    {
        $feed = $this->resyncManagerPool->getActualFeedName($input->getOption(ResyncOptions::FEED_OPTION));

        foreach ($this->resyncOptions->getOptionsList() as $optionConfig) {
            if ($value = $input->getOption($optionConfig['name'])) {
                $this->resyncOptions->setOptionValue(
                    $optionConfig['name'],
                    $value
                );
            }
        }
        $this->configOptionsHandler->initialize($this->resyncOptions->getOptionValues());

        if (isset($this->feedNames[$feed])) {
            $this->progressBarManager->setOutput($output);
            $feedName = $this->feedNames[$feed];
            $this->resyncManager = $this->resyncManagerPool->getResyncManager($feed);
            if ($input->getOption(ResyncOptions::NO_REINDEX_OPTION)) {
                try {
                    $startTime = microtime(true);
                    $output->writeln('<info>Re-submitting ' . $feedName . ' feed data...</info>');
                    $this->resyncManager->executeResubmitOnly();
                    $time = $this->formatTime((int)(microtime(true) - $startTime));
                    $output->writeln('');
                    $output->writeln('<info>' . $feedName . ' feed data re-submit complete in ' . $time .'</info>');
                    $returnStatus = Cli::RETURN_SUCCESS;
                } catch (\Exception $ex) {
                    $output->writeln(
                        '<error>An error occurred re-submitting ' . $feedName . ' feed data to SaaS service.</error>'
                    );
                    $returnStatus = Cli::RETURN_FAILURE;
                }
            } else {
                try {
                    $startTime = microtime(true);
                    $output->writeln('<info>Executing full re-sync of ' . $feedName . ' feed data...</info>');
                    $this->resyncManager->executeFullResync();
                    $time = $this->formatTime((int)(microtime(true) - $startTime));
                    $output->writeln('');
                    $output->writeln('<info>' . $feedName . ' feed data full re-sync complete in ' . $time .'</info>');
                    $returnStatus = Cli::RETURN_SUCCESS;
                } catch (\Exception $ex) {
                    $output->writeln('<error>An error occurred re-syncing ' . $feedName
                        . ' feed data to SaaS service: ' . $ex->getMessage() .'.</error>');
                    $returnStatus = Cli::RETURN_FAILURE;
                }
            }
        } else {
            $output->writeln(
                '<error>Resync option --feed is required. Available feeds: '
                . implode(', ', array_keys($this->feedNames))
                . '</error>'
            );
            $returnStatus = Cli::RETURN_FAILURE;
        }

        return $returnStatus;
    }

    /**
     * Format time in seconds to days, hours, minutes and seconds
     *
     * @param int $seconds
     * @return string
     */
    private function formatTime(int $seconds): string
    {
        static $secondsInDay = 86400;
        $daysCount = 0;
        $seconds = $seconds - $secondsInDay;
        while ($seconds >= 0) {
            $daysCount++;
            $seconds = $seconds - $secondsInDay;
        }
        $seconds = $seconds + $secondsInDay;
        $time = gmdate('H:i:s', $seconds);
        return $daysCount > 0 ? sprintf('%s day(s) %s', $daysCount, $time) : $time;
    }
}
