<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\SaaSCatalog\Cron;

use Magento\CatalogDataExporter\Model\Feed\Products as ProductsFeed;
use Magento\DataExporter\Exception\UnableSendData;
use Magento\DataExporter\Http\Command\SubmitFeed;
use Magento\Framework\FlagManager;
use Magento\Framework\Module\ModuleList;
use Magento\SaaSCatalog\Model\ProductFeedRegistry;
use Psr\Log\LoggerInterface;

/**
 * Class SubmitProductFeed
 */
class SubmitProductFeed
{
    /**
     * @var SubmitFeed
     */
    private $submitFeed;

    /**
     * @var ModuleList
     */
    private $moduleList;
    /**
     * @var ProductsFeed
     */
    private $productsFeed;

    /**
     * @var FlagManager
     */
    private $flagManager;

    /**
     * @var ProductFeedRegistry
     */
    private $feedRegistry;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var int
     */
    private $chunkSize;

    /**
     * SubmitProductFeed constructor.
     *
     * @param ProductsFeed $productsFeed
     * @param SubmitFeed $submitFeed
     * @param ModuleList $moduleList
     * @param FlagManager $flagManager
     * @param ProductFeedRegistry $feedRegistry
     * @param LoggerInterface $logger
     * @param int $chunkSize
     */
    public function __construct(
        ProductsFeed $productsFeed,
        SubmitFeed $submitFeed,
        ModuleList $moduleList,
        FlagManager $flagManager,
        ProductFeedRegistry $feedRegistry,
        LoggerInterface $logger,
        int $chunkSize = 100
    ) {
        $this->productsFeed = $productsFeed;
        $this->submitFeed = $submitFeed;
        $this->moduleList = $moduleList;
        $this->flagManager = $flagManager;
        $this->feedRegistry = $feedRegistry;
        $this->logger = $logger;
        $this->chunkSize = $chunkSize;
    }

    /**
     * @param array $data
     * @return bool
     * @throws UnableSendData
     */
    public function submitFeed(array $data) : bool
    {
        $chunks = array_chunk($data['feed'], $this->chunkSize);
        $result = true;
        foreach ($chunks as $chunk) {
            $filteredData = $this->feedRegistry->filter($chunk);
            if (!empty($filteredData)) {
                $result = $this->submitFeed->execute(
                    'products',
                    $filteredData,
                    [
                        sprintf(
                            'module-version:%s',
                            $this->moduleList->getOne('Magento_CatalogDataExporter')['setup_version']
                        )
                    ]
                );
                if (!$result) {
                    return $result;
                } else {
                    $this->feedRegistry->registerFeed($filteredData);
                }
            }
        }
        return $result;
    }

    /**
     * Site verification and claiming
     */
    public function execute()
    {
        $lastSyncTimestamp = $this->flagManager->getFlagData('products-feed-version');
        $data = $this->productsFeed->getFeedSince($lastSyncTimestamp ? $lastSyncTimestamp : '1');
        try {
            if ($data['recentTimestamp'] !== null) {
                $result = $this->submitFeed($data);
                if ($result) {
                    $this->flagManager->saveFlag('products-feed-version', $data['recentTimestamp']);
                }
            }
        } catch (\Exception $exception) {
            $this->logger->error($exception->getMessage());
        }
    }
}
