<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\QuickCheckout\Model;

use Magento\Checkout\Model\ConfigProviderInterface;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\Framework\Locale\ResolverInterface;

/**
 * Provide configuration for checkout components
 *
 * @SuppressWarnings(PHPMD.CookieAndSessionMisuse)
 */
class ConfigProvider implements ConfigProviderInterface
{
    public const CODE = 'quick_checkout';

    /**
     * @var Config
     */
    private $config;

    /**
     * @var ResolverInterface
     */
    private $localeResolver;

    /**
     * @var CustomerSession
     */
    private $customerSession;

    /**
     * Config provider class constructor
     *
     * @param Config $config
     * @param ResolverInterface $localeResolver
     * @param CustomerSession $customerSession
     */
    public function __construct(
        Config $config,
        ResolverInterface $localeResolver,
        CustomerSession $customerSession
    ) {
        $this->config = $config;
        $this->localeResolver = $localeResolver;
        $this->customerSession = $customerSession;
    }

    /**
     * @inheritdoc
     */
    public function getConfig(): array
    {
        if (!$this->config->isEnabled()) {
            return [];
        }

        return [
            'payment' => [
                self::CODE => [
                    'publishableKey' => $this->config->getPublishableKey(),
                    'locale' => str_replace('_', '-', $this->localeResolver->getLocale()),
                    'canDisplayOtpPopup' => $this->config->canDisplayOtpPopup(),
                    'creditCardComponentConfig' => $this->config->getCreditCardFormConfig(),
                    'isLoggedInBolt' => !empty($this->customerSession->getBoltCustomerToken()),
                    'canUseSso' => $this->customerSession->getCanUseBoltSso(),
                    'canTrackCheckout' => $this->config->isCheckoutTrackingEnabled(),
                ]
            ]
        ];
    }
}
