<?php

/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\QuickCheckout\Model;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Module\Manager;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;

/**
 * Payment method config
 */
class Config
{
    public const ENVIRONMENT_SANDBOX = 'sandbox';

    public const ENVIRONMENT_PRODUCTION = 'production';

    private const SDK_HOST_SANDBOX = 'connect-sandbox.bolt.com';

    private const SDK_HOST_PRODUCTION = 'connect.bolt.com';

    private const API_HOST_SANDBOX = 'api-sandbox.bolt.com';

    private const API_HOST_PRODUCTION = 'api.bolt.com';

    /**
     * @var ScopeConfigInterface
     */
    private $scopeConfig;

    /** @var Manager */
    private $moduleManager;

    /**
     * @var StoreManagerInterface
     */
    private $storeManager;

    /**
     * @var array
     */
    private $creditCardComponentConfig;

    /**
     * @param ScopeConfigInterface $scopeConfig
     * @param Manager $moduleManager
     * @param StoreManagerInterface $storeManager
     * @param array $creditCardComponentConfig
     */
    public function __construct(
        ScopeConfigInterface $scopeConfig,
        Manager $moduleManager,
        StoreManagerInterface $storeManager,
        array $creditCardComponentConfig = []
    ) {
        $this->scopeConfig = $scopeConfig;
        $this->moduleManager = $moduleManager;
        $this->storeManager = $storeManager;
        $this->creditCardComponentConfig = $creditCardComponentConfig;
    }

    /**
     * Is enabled?
     *
     * @param string|null $code
     * @return bool
     */
    public function isEnabled(string $code = null): bool
    {
        $path = 'payment/quick_checkout/active';
        if ($code == null) {
            return (bool)$this->getConfigValue($path);
        }
        return (bool)$this->getConfigValue($path, ScopeInterface::SCOPE_WEBSITE, $code);
    }

    /**
     * Can display OTP popup?
     *
     * @return bool
     */
    public function canDisplayOtpPopup(): bool
    {
        return !$this->getConfigValue('carriers/instore/active') && !empty($this->getPublishableKey());
    }

    /**
     * Get SDK URL
     *
     * @return string
     */
    public function getSdkUrl(): string
    {
        return 'https://' . $this->getSdkHost() . '/embed-accounts.js';
    }

    /**
     * Get host
     *
     * @return string
     */
    private function getSdkHost(): string
    {
        return $this->isProductionEnvironment() ? self::SDK_HOST_PRODUCTION : self::SDK_HOST_SANDBOX;
    }

    /**
     * Get host
     *
     * @return string
     */
    public function getApiHost(): string
    {
        return $this->isProductionEnvironment() ? self::API_HOST_PRODUCTION : self::API_HOST_SANDBOX;
    }

    /**
     * Get bolt api key
     *
     * @param string|null $code
     * @return string|null
     */
    public function getApiKey(string $code = null): ?string
    {
        $path = 'payment/quick_checkout/api_key';
        if ($code == null) {
            return $this->getConfigValue($path);
        }
        return $this->getConfigValue($path, ScopeInterface::SCOPE_WEBSITE, $code);
    }

    /**
     * Get bolt signing secret
     *
     * @return string|null
     */
    public function getSigningSecret(): ?string
    {
        return $this->getConfigValue('payment/quick_checkout/signing_secret');
    }

    /**
     * Get bolt publishable key
     *
     * @param string|null $code
     * @return string|null
     */
    public function getPublishableKey(string $code = null): ?string
    {
        $path = 'payment/quick_checkout/publishable_key';
        if ($code == null) {
            return $this->getConfigValue($path);
        }
        return $this->getConfigValue($path, ScopeInterface::SCOPE_WEBSITE, $code);
    }

    /**
     * Get configs for credit card form
     *
     * @return array
     */
    public function getCreditCardFormConfig(): array
    {
        return $this->creditCardComponentConfig;
    }

    /**
     * Get config value
     *
     * @param string $path
     * @param string $scope
     * @param string|null $code
     * @return string|null
     */
    private function getConfigValue(
        string $path,
        string $scope = ScopeInterface::SCOPE_STORE,
        string $code = null
    ): ?string {
        return $this->scopeConfig->getValue(
            $path,
            $scope,
            $code
        );
    }

    /**
     * Is production environment?
     *
     * @param string|null $code
     * @return bool
     */
    public function isProductionEnvironment(string $code = null): bool
    {
        $path = 'payment/quick_checkout/method';
        if ($code == null) {
            return $this->getConfigValue($path) == self::ENVIRONMENT_PRODUCTION;
        } else {
            return $this->getConfigValue($path, ScopeInterface::SCOPE_WEBSITE, $code) == self::ENVIRONMENT_PRODUCTION;
        }
    }

    /**
     * Returns true if the admin usage tracking is enabled
     *
     * @return bool
     */
    public function isAdminUsageEnabled(): bool
    {
        $path = 'admin/usage/enabled';

        if (!$this->moduleManager->isEnabled('Magento_AdminAnalytics')) {
            return false;
        }

        return (bool)$this->getConfigValue($path) === true;
    }

    /**
     * Returns true if the extension is active in any of the available stores
     *
     * @return bool
     */
    public function hasConfiguredKeys(): bool
    {
        foreach ($this->storeManager->getWebsites() as $website) {
            if ($this->isEnabled($website->getCode())) {
                return true;
            }
        }
        return false;
    }
}
