/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */

define([
    'jquery',
    'underscore',
    'ko',
    'mageUtils',
    'Magento_Checkout/js/view/shipping-address/list',
    'uiLayout',
    'Magento_QuickCheckout/js/model/customer/address-list',
    'Magento_QuickCheckout/js/model/account-handling',
    'Magento_QuickCheckout/js/model/customer/customer',
    'Magento_Checkout/js/model/shipping-address/form-popup-state',
    'Magento_Checkout/js/checkout-data'
], function (
    $,
    _,
    ko,
    utils,
    ShippingAddressListComponent,
    layout,
    addressList,
    bolt,
    boltCustomer,
    formPopupState,
    checkoutData
) {
    'use strict';

    var defaultRendererTemplate = {
        parent: '${ $.$data.parentName }',
        name: '${ $.$data.name }',
        component: 'Magento_QuickCheckout/js/view/shipping-address/address-renderer/default',
        provider: 'checkoutProvider'
    };

    return ShippingAddressListComponent.extend({
        defaults: {
            template: 'Magento_QuickCheckout/shipping-address/list',
            shippingFormTemplate: 'Magento_QuickCheckout/shipping-address/form',
            rendererTemplates: [],
            rendererComponents: [],
            shippingFormSelector: '#co-shipping-form',
            shippingStepSelector: '#checkout-step-shipping',
            saveAddressToBoltCheckboxSelector: '#shipping-save-to-bolt',
            boltListVisible: ko.observable(addressList().length > 0),
            shouldSaveAddress: ko.observable(checkoutData.getIsSaveNewAddress())
        },

        /**
         * @inheritdoc
         */
        initialize: function () {
            var self = this;

            this._super();
            this.initChildren();

            addressList.subscribe(function (changes) {
                    changes.forEach(function (change) {
                        if (change.status === 'added') {
                            self.createRendererComponent(change.value, change.index);
                        }
                    });
                },
                this,
                'arrayChange'
            );

            boltCustomer.isBoltUser.subscribe(function () {
                if (boltCustomer.isBoltUser()) {
                    addressList(bolt.getBoltShippingAddresses());
                    if (addressList().length > 0) {
                        $(self.shippingFormSelector).remove();
                    }
                }
                self.boltListVisible(boltCustomer.isBoltUser() && addressList().length > 0);
            });

            return this;
        },

        /**
         * @inheritdoc
         */
        initChildren: function () {
            _.each(addressList(), this.createRendererComponent, this);
            return this;
        },

        /**
         * Create new component that will render given address in the address list
         *
         * @param {Object} address
         * @param {*} index
         */
        createRendererComponent: function (address, index) {
            var rendererTemplate,
                templateData,
                rendererComponent;

            if (index in this.rendererComponents) {
                this.rendererComponents[index].address(address);
            } else {
                // rendererTemplates are provided via layout
                rendererTemplate = address.getType() != undefined && this.rendererTemplates[address.getType()] != undefined ? //eslint-disable-line
                    utils.extend({}, defaultRendererTemplate, this.rendererTemplates[address.getType()]) :
                    defaultRendererTemplate;
                templateData = {
                    parentName: this.name,
                    name: index
                };
                rendererComponent = utils.template(rendererTemplate, templateData);
                utils.extend(rendererComponent, {
                    address: ko.observable(address)
                });
                layout([rendererComponent]);
                this.rendererComponents[index] = rendererComponent;
            }
        },

        /**
         * Set state of address popup
         */
        openParentPopup: function () {
            formPopupState.isVisible(true);
        },

        /**
         * Binding click function for save address checkbox state
         */
        saveAddressToBolt: function () {
            var checked = $(this.saveAddressToBoltCheckboxSelector).prop('checked');

            this.shouldSaveAddress(checked);
            checkoutData.setIsSaveNewAddress(checked);
        },

        /**
         * Handle standard shipping forms after rendering
         */
        afterRender: function () {
            var self = this,
                observer = new MutationObserver(function () {
                    // Remove default Magento shipping address form on checkout reload if bolt user
                    if (boltCustomer.isBoltUser() && addressList().length > 0) {
                        $(self.shippingFormSelector).remove();
                    }
                });

            observer.disconnect();
            observer.observe($(self.shippingStepSelector).get(0), {
                attributes: true,
                subtree: true
            });
        }
    });
});
