<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\QuickCheckout\Model;

use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Framework\App\Request\Http;
use \InvalidArgumentException;
use Magento\Payment\Gateway\Http\ClientException;
use Magento\Payment\Gateway\Http\ClientInterface;
use Magento\Payment\Gateway\Http\ConverterException;
use Magento\QuickCheckout\Gateway\Http\TransferFactory;

/**
 * Service class to add new payment methods to bolt wallet
 * @SuppressWarnings(PHPMD.CookieAndSessionMisuse)
 */
class PaymentMethodService
{
    /**
     * @var TransferFactory
     */
    private $transferFactory;

    /**
     * @var ClientInterface
     */
    private $serviceClient;

    /**
     * @var CheckoutSession
     */
    private $checkoutSession;

    /**
     * @param TransferFactory $transferFactory
     * @param ClientInterface $serviceClient
     * @param CheckoutSession $checkoutSession
     */
    public function __construct(
        TransferFactory $transferFactory,
        ClientInterface $serviceClient,
        CheckoutSession $checkoutSession
    ) {
        $this->transferFactory = $transferFactory;
        $this->serviceClient = $serviceClient;
        $this->checkoutSession = $checkoutSession;
    }

    /**
     * Add payment method to bolt wallet
     *
     * @param array $paymentMethodCardInformation
     * @return array
     * @throws ClientException
     * @throws ConverterException
     * @throws InvalidArgumentException
     */
    public function addPaymentMethod(array $paymentMethodCardInformation): array
    {
        $token = $this->checkoutSession->getData('bolt_customer_token');
        if (!$token) {
            throw new InvalidArgumentException('No valid bolt session token available');
        }
        $request = [
            'uri' => '/v1/account/payment_methods',
            'method' => Http::METHOD_POST,
            'body' => $paymentMethodCardInformation,
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'bearer ' . $token,
            ]
        ];
        $transferObject = $this->transferFactory->create($request);
        return $this->serviceClient->placeRequest($transferObject);
    }
}
