<?php
/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\QuickCheckout\Model;

use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Framework\App\Request\Http;
use \InvalidArgumentException;
use Magento\Payment\Gateway\Http\ClientException;
use Magento\Payment\Gateway\Http\ClientInterface;
use Magento\Payment\Gateway\Http\ConverterException;
use Magento\QuickCheckout\Gateway\Http\TransferFactory;

/**
 * Service class to add new addresses to bolt account
 * @SuppressWarnings(PHPMD.CookieAndSessionMisuse)
 */
class AddressService
{
    /**
     * @var TransferFactory
     */
    private $transferFactory;

    /**
     * @var ClientInterface
     */
    private $serviceClient;

    /**
     * @var CheckoutSession
     */
    private $checkoutSession;

    /**
     * @param TransferFactory $transferFactory
     * @param ClientInterface $serviceClient
     * @param CheckoutSession $checkoutSession
     */
    public function __construct(
        TransferFactory $transferFactory,
        ClientInterface $serviceClient,
        CheckoutSession $checkoutSession
    ) {
        $this->transferFactory = $transferFactory;
        $this->serviceClient = $serviceClient;
        $this->checkoutSession = $checkoutSession;
    }

    /**
     * Add address to bolt wallet
     *
     * @param array $addressInformation
     * @return array
     * @throws ClientException
     * @throws ConverterException
     * @throws InvalidArgumentException
     */
    public function addAddress(array $addressInformation): array
    {
        $token = $this->checkoutSession->getData('bolt_customer_token');
        if (!$token) {
            throw new InvalidArgumentException('No valid bolt session token available');
        }
        $request = [
            'uri' => '/v1/account/addresses',
            'method' => Http::METHOD_POST,
            'body' => $addressInformation,
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => 'bearer ' . $token,
            ]
        ];
        $transferObject = $this->transferFactory->create($request);
        return $this->serviceClient->placeRequest($transferObject);
    }
}
