<?php

/**
 * Copyright © Magento, Inc. All rights reserved.
 * See COPYING.txt for license details.
 */
declare(strict_types=1);

namespace Magento\QuickCheckoutAdminPanel\Model\Reporting;

use DateTime;
use InvalidArgumentException;

class Filters
{
    private const DATE_FORMAT = 'Y-m-d';

    /**
     * @var string
     */
    private string $startDate;

    /**
     * @var string
     */
    private string $endDate;

    /**
     * @var int|null
     */
    private ?int $websiteId;

    /**
     * @var string|null
     */
    private ?string $device;

    /**
     * @param string $startDate
     * @param string $endDate
     * @param int|null $websiteId
     * @param string|null $device
     */
    public function __construct(
        string $startDate,
        string $endDate,
        ?int $websiteId,
        ?string $device
    ) {
        $parsedStartDate = $this->parseDatetime($startDate);
        $parsedEndDate = $this->parseDatetime($endDate);
        $this->assertValidRange($parsedStartDate, $parsedEndDate);

        $this->startDate = $startDate;
        $this->endDate = $endDate;
        $this->websiteId = $websiteId;
        $this->device = $device;
    }

    /**
     * Creates a DateTime object from the given date string
     *
     * @param string $datetime
     * @return DateTime
     */
    private function parseDatetime(string $datetime): DateTime
    {
        // @phpstan-ignore-next-line
        $dateTime = DateTime::createFromFormat(self::DATE_FORMAT, $datetime);
        if (!$dateTime) {
            throw new InvalidArgumentException('Invalid datetime');
        }
        return $dateTime;
    }

    /**
     * Validates that the start date is older than the end date
     *
     * @param DateTime $startDate
     * @param DateTime $endDate
     * @return void
     */
    private function assertValidRange(DateTime $startDate, DateTime $endDate): void
    {
        if ($startDate > $endDate) {
            throw new InvalidArgumentException('Start date must be older than the end date');
        }
    }

    /**
     * Returns the start date
     *
     * @return string
     */
    public function getStartDate(): string
    {
        return $this->startDate;
    }

    /**
     * Returns the end date
     *
     * @return string
     */
    public function getEndDate(): string
    {
        return $this->endDate;
    }

    /**
     * Returns the website id
     *
     * @return int|null
     */
    public function getWebsiteId(): ?int
    {
        return $this->websiteId;
    }

    /**
     * Returns the device
     *
     * @return string|null
     */
    public function getDevice(): ?string
    {
        return $this->device;
    }
}
